/*
  WriteVoltage

  Reads an analog voltage from pin 0, and writes it to a channel on ThingSpeak every 20 seconds.

  ThingSpeak ( https://www.thingspeak.com ) is an analytic IoT platform service that allows you to aggregate, visualize and
  analyze live data streams in the cloud.

  Copyright 2017, The MathWorks, Inc.

  Documentation for the ThingSpeak Communication Library for Arduino is in the extras/documentation folder where the library was installed.
  See the accompaning licence file for licensing information.
*/

#include "FishinoThingSpeak.h"

// ***********************************************************************************************************
// This example selects the correct library to use based on the board selected under the Tools menu in the IDE.
// Yun, Ethernet shield, WiFi101 shield, esp8266, and MXR1000 are all supported.
// With Yun, the default is that you're using the Ethernet connection.
// If you're using a wi-fi 101 or ethernet shield (http://www.arduino.cc/en/Main/ArduinoWiFiShield), uncomment the corresponding line below
// ***********************************************************************************************************

//#define USE_WIFI101_SHIELD
//#define USE_ETHERNET_SHIELD

#if !defined(USE_WIFI101_SHIELD) && !defined(USE_ETHERNET_SHIELD) && !defined(ARDUINO_SAMD_MKR1000) && \
    !defined(ARDUINO_AVR_YUN) && !defined(ARDUINO_ARCH_ESP8266) && !defined(FISHINO_WIFI)
#error "Uncomment the #define for either USE_WIFI101_SHIELD or USE_ETHERNET_SHIELD"
#endif

#if defined(ARDUINO_AVR_YUN)
	#include "YunClient.h"
	YunClient client;
#elif defined(FISHINO_WIFI)
	#include <Fishino.h>
	char ssid[] = "<YOURNETWORK>";    //  your network SSID (name)
	char pass[] = "<YOURPASSWORD>";   // your network password
	int status = STATION_IDLE;
	FishinoClient  client;
#elif defined(USE_WIFI101_SHIELD) || defined(ARDUINO_SAMD_MKR1000) || defined(ARDUINO_ARCH_ESP8266)
	// Use WiFi
	#ifdef ARDUINO_ARCH_ESP8266
		#include <ESP8266WiFi.h>
	#else
		#include <SPI.h>
		#include <WiFi101.h>
	#endif
	char ssid[] = "<YOURNETWORK>";    //  your network SSID (name)
	char pass[] = "<YOURPASSWORD>";   // your network password
	int status = WL_IDLE_STATUS;
	WiFiClient  client;
#elif defined(USE_ETHERNET_SHIELD)
	// Use wired ethernet shield
	#include <SPI.h>
	#include <Ethernet.h>
	byte mac[] = { 0xDE, 0xAD, 0xBE, 0xEF, 0xFE, 0xED};
	EthernetClient client;
#endif

#ifdef ARDUINO_ARCH_AVR
	// On Arduino:  0 - 1023 maps to 0 - 5 volts
	#define VOLTAGE_MAX 5.0
	#define VOLTAGE_MAXCOUNTS 1023.0
#elif defined(_FISHINO_PIC32_)
	// On FISHINO32:  0 - 1023 maps to 0 - 3.3 volts
	#define VOLTAGE_MAX 3.3
	#define VOLTAGE_MAXCOUNTS 1023.0
#elif ARDUINO_SAMD_MKR1000
	// On MKR1000:  0 - 1023 maps to 0 - 3.3 volts
	#define VOLTAGE_MAX 3.3
	#define VOLTAGE_MAXCOUNTS 1023.0
#elif ARDUINO_SAM_DUE
	// On Due:  0 - 1023 maps to 0 - 3.3 volts
	#define VOLTAGE_MAX 3.3
	#define VOLTAGE_MAXCOUNTS 1023.0
#elif ARDUINO_ARCH_ESP8266
	// On ESP8266:  0 - 1023 maps to 0 - 1 volts
	#define VOLTAGE_MAX 1.0
	#define VOLTAGE_MAXCOUNTS 1023.0
#endif

/*
  *****************************************************************************************
  **** Visit https://www.thingspeak.com to sign up for a free account and create
  **** a channel.  The video tutorial http://community.thingspeak.com/tutorials/thingspeak-channels/
  **** has more information. You need to change this to your channel, and your write API key
  **** IF YOU SHARE YOUR CODE WITH OTHERS, MAKE SURE YOU REMOVE YOUR WRITE API KEY!!
  *****************************************************************************************/
unsigned long myChannelNumber = 31461;
const char * myWriteAPIKey = "LD79EOAAWRVYF04Y";

void setup()
{

#ifdef ARDUINO_AVR_YUN
	Bridge.begin();
#elif defined(FISHINO_WIFI)
	Fishino.begin(ssid, pass);
#elif defined(ARDUINO_ARCH_ESP8266) || defined(USE_WIFI101_SHIELD) || defined(ARDUINO_SAMD_MKR1000)
	WiFi.begin(ssid, pass);
#else
	Ethernet.begin(mac);
#endif

	ThingSpeak.begin(client);
}

void loop()
{
	// read the input on analog pin 0:
	int sensorValue = analogRead(A0);
	// Convert the analog reading
	// On Uno,Mega,YunArduino:  0 - 1023 maps to 0 - 5 volts
	// On ESP8266:  0 - 1023 maps to 0 - 1 volts
	// On MKR1000,Due: 0 - 4095 maps to 0 - 3.3 volts
	float voltage = sensorValue * (VOLTAGE_MAX / VOLTAGE_MAXCOUNTS);

	// Write to ThingSpeak. There are up to 8 fields in a channel, allowing you to store up to 8 different
	// pieces of information in a channel.  Here, we write to field 1.
	ThingSpeak.writeField(myChannelNumber, 1, voltage, myWriteAPIKey);
	delay(20000); // ThingSpeak will only accept updates every 15 seconds.
}
