//////////////////////////////////////////////////////////////////////////
//		FishBook - Capitolo 16 - Suoniamo una melodia					//
//	Generiamo una melodia facilmente modificabile tramite una stringa	//
//////////////////////////////////////////////////////////////////////////

// la melodia da suonare
// love me tender
const char *melodia = 
	"DO4D2P10,FA4D2P10,MI4D2P10,FA4D2P10,SOL4D2P10,RE4D2P10,SOL4D4P10,FA4D2P10,MI4D2P10,RE4D3P10,MI4D1P10,FA4D8P10,"
	"DO4D2P10,FA4D2P10,MI4D2P10,FA4D2P10,SOL4D2P10,RE4D2P10,SOL4D4P10,FA4D2P10,MI4D2P10,RE4D3P10,MI4D1P10,FA4D8P10,"
	"LA4D2P10,LA4D2P10,LA4D2P10,LA4D2P10,LA4D2P10,LA4D2P10,LA4D4P10,LA4D2P10,SOL4D2P10,FA4D3P10,SOL4D1P10,LA4D8P10,"
	"LA4D2P10,LA4D2P10,LA#4D2P10,LA4D2P10,SOL4D2P10,RE4D2P10,SOL4D4P10,FA4D2P10,MI4D2P10,LA4D3P10,SOL4D1P10,FA4D8P10"
;


// sequenza di note in un'ottava
// l'eventuale D finale sta per DIESIS
enum NOTE
{
	DO, DOD, RE, RED, MI, FA, FAD, SOL, SOLD, LA, LAD, SI
};

// array di stringhe contenenti i nomi delle note
const char *note[] = { "DO", "DO#", "RE", "RE#", "MI", "FA", "FA#", "SOL", "SOL#", "LA", "LA#", "SI" };

// funzione per determinare la frequenza data la nota e l'ottava
// vogliamo evitare di inserire tabelle sterminate nel codice!
int frequenza(int nota, int ottava)
{
	// partiamo dal LA centrale, a 440 Hz (quarta ottava)
	#define LA_4  440
	
	// l'esponente di 2 per ricavare la nota partendo dal LA della quarta ottava
	double esponente = ((double)ottava - 4) + ((double)nota - (LA - DO)) / 12.0;
	
	// calcola la frequenza della nota scelta
	double freq = (double)LA_4 * pow(2.0, esponente);
	
	
	// arrotonda la frequenza al numero intero più vicino e la ritorna
	return (int)(freq + 0.5);

}

// confronta 2 stringhe di lunghezza massima data ignorando maiuscole/minuscole
// dovrebbe essere una funzione standard del C, ma nelle librerie manca
int strnicmp(const char *s1, const char *s2, int n)
{
	for(int i = 0; i < n; i++)
	{
		// se una delle due stringhe è troppo corta non sono uguali
		if(!*s1 || !*s2)
			return -1;
		
		// se il carattere corrente è diverso, non sono uguali
		if(toupper(*s1++) != toupper(*s2++))
			return -1;
	}
	
	// tutti i caratteri sono uguali, ritorna 0
	return 0;
}

// funzione per estrarre la nota dalla stringa
// parametro : puntatore a puntatore a carattere
// restituisce il numero della nota, o -1 in caso di errore
// il parametro viene avanzato ad inizio ottava
int leggiNota(const char **ptr)
{
	const char *inizio = *ptr;
	const char *fine = inizio;
	
	// avanza con 'fine' finchè trova una lettera o un #
	while(isalpha(*fine) || *fine == '#')
		fine++;
	
	// determina la lunghezza del nome
	int len = fine - inizio;
	
	// aggiorna il puntatore alla stringa di note
	*ptr = fine;
	
	// cerca il nome nell'array di nomi
	for(int i = 0; i < 12; i++)
		if(!strnicmp(inizio, note[i], len))
			return i;
		
	// non trovata, ritorna errore
	return -1;
}

// funzione per estrarre l'ottava dalla stringa
// parametro : puntatore a puntatore a carattere
// restituisce il numero dell'ottava, o -1 in caso di errore
// il parametro viene avanzato ad inizio durata
int leggiOttava(const char **ptr)
{
	const char *inizio = *ptr;
	
	// il primo carattere dell'ottava DEVE essere una cifra
	if(!isdigit(*inizio))
		return -1;
	
	// inizializza l'ottava
	int ottava = 0;
	
	// legge il numero fornito cifra per cifra ed aggiorna l'ottava
	while(isdigit(*inizio))
		ottava = ottava * 10 + *inizio++ - '0';

	// aggiorna il puntatore alla stringa di note
	*ptr = inizio;
	
	// ritorna l'ottava
	return ottava;
}

// funzione per estrarre la durata dalla stringa
// parametro : puntatore a puntatore a carattere
// restituisce la durata (in multipli di 1/8), o -1 in caso di errore
// il parametro viene avanzato ad inizio pausa
int leggiDurata(const char **ptr)
{
	const char *inizio = *ptr;
	
	// la durata deve iniziare con la lettera D
	if(toupper(*inizio++) != 'D')
		return -1;
	
	// inizializza la durata
	int durata = 0;
	
	// legge il numero fornito cifra per cifra ed aggiorna la durata
	while(isdigit(*inizio))
		durata = durata * 10 + *inizio++ - '0';

	// aggiorna il puntatore alla stringa di note
	*ptr = inizio;
	
	// ritorna la durata
	return durata;
}

// funzione per estrarre la pausa dalla stringa
// parametro : puntatore a puntatore a carattere
// restituisce la pausa (in millisecondi), o -1 in caso di errore
// il parametro viene avanzato ad inizio prossima nota
int leggiPausa(const char **ptr)
{
	const char *inizio = *ptr;
	
	// la pausa deve iniziare con la lettera P
	if(toupper(*inizio++) != 'P')
		return -1;
	
	// inizializza la pausa
	int pausa = 0;
	
	// legge il numero fornito cifra per cifra ed aggiorna la pausa
	while(isdigit(*inizio))
		pausa = pausa * 10 + *inizio++ - '0';

	// aggiorna il puntatore alla stringa di note
	*ptr = inizio;
	
	// ritorna la pausa
	return pausa;
}

// segnala errore e blocca lo sketch
void stop(const char *msg, int pos)
{
	Serial.println(msg);
	Serial.print("Alla posizione ");
	Serial.println(pos);
	while(true)
		;
}

void setup(void)
{
	Serial.begin(115200);
	while(!Serial)
		;
}

// ciclo infinito
void loop(void)
{
	// punta alla prima nota della sequenza
	const char *ptr = melodia;
	
	// finchè ci sono note...
	while(*ptr)
	{
		int nota = leggiNota(&ptr);
		if(nota == -1)
			stop("Errore lettura nota", ptr - melodia);

		int ottava = leggiOttava(&ptr);
		if(ottava == -1)
			stop("Errore lettura ottava", ptr - melodia);
		
		int durata = leggiDurata(&ptr);
		if(durata == -1)
			stop("Errore lettuta durata", ptr - melodia);
		
		int pausa = leggiPausa(&ptr);
		if(pausa == -1)
			stop("Errore lettura pausa", ptr - melodia);
		
		// controlla che ci sia una virgola per separare dalla prossima nota
		// oppure che la stringa sia finita
		if(*ptr && *ptr != ',')
			stop("Errore, manca virgola di separazione note", ptr - melodia);
		
		// avanza alla prossima nota, se c'è
		if(*ptr)
			ptr++;
		
		// calcola la frequenza da suonare
		int freq = frequenza(nota, ottava);
		
		// calcola la durata in millisecondi
		durata = 250 * durata;
		
		// suona la nota corrente
		tone(9, freq, durata);
		
		// attende per un tempo durata + pausa
		delay(durata + pausa);
		
		noTone(3);
	}
	
	// mezzo secondo tra le ripetizioni della melodia
	delay(500);
}
