//////////////////////////////////////////////////////////////////////////
//	FishBook - Capitolo 21 - Registrazione di data, ora e temperatura	//
//	Completiamo il nostro orologio/datario/termometro con la			//
//	registrazione dei dati su una scheda microSD						//
//////////////////////////////////////////////////////////////////////////

#include <LiquidCrystal.h>
#include <Wire.h>
#include <RTClib.h>
#include <OneWire.h>
#include <DallasTemperature.h>
#include <SD.h>

// il display LCD
LiquidCrystal LCD(2, 3, 5, 6, 8, 9);

// l' RTC
RTC_DS1307 rtc;

// i nomi dei mesi
const char *nomiMesi[] = {
	"Gen", "Feb", "Mar", "Apr", "Mag", "Giu",
	"Lug", "Ago", "Set", "Ott", "Nov", "Dic"
};

// imposta l'interfaccia OneWire per comunicare sul pin 14 (alias A0)
OneWire oneWire(14);

// la variabile per il sensore di temperatura
DallasTemperature tempSensors(&oneWire);

// flag che indica se la scheda SD è a posto e funzionante
bool sdOK;

// flag per far lampeggiare la scritta LOG o NOSD
bool blink;

// il nome del file di log
const char *logName = "templog.txt";

// intervallo in secondi tra 2 logs
const int logInterval = 10;

// la temperatura letta dal sensore
double temp;

// vogliamo leggere l'ora ogni secondo; invece di usare una delay()
// che non tiene conto del tempo perso dal programma, usiamo la
// millis() con una variabile
uint32_t tim;

// i "ticks" tra una lettura di temperatura e l'altra
// ad ogni tick corrisponde un secondo circa, quindi
// leggiamo la temperatura ogni 10 ticks visto che la lettura
// dal sensore porta via parecchio tempo!
int tempTicks;

// i "ticks" tra un log e l'altro, in modo da poter specificare
// ogni quanti secondi eseguire un log
int logTicks;

void setup(void)
{
	Serial.begin(115200);
	
	// inizializza l'LCD
	LCD.begin(16, 2);
	
	// inizializza la libreria I2C
	Wire.begin();
	
	// inizializza l' RTC
	rtc.begin();

	// se l'RTC ha perso l'ora (o se è al primo utilizzo
	// imposta l'ora a quella di compilazione dello sketch
	if (! rtc.isrunning())
		rtc.adjust(DateTime(F(__DATE__), F(__TIME__)));
	
	// inizializza il sensore di temperatura
	tempSensors.begin();
	
	sdOK = true;
	
	
	// controlla che la scheda SD sia presente e operativa
	// 4 è il pin del Chip Select per la scheda SD
	if (!SD.begin(4))
		sdOK = false;
	
	// controlla la possibilità di aprire il file di log
	// sulla SD; se impossibile disabilita il log
	if(sdOK)
	{
		// apre/crea il file
		File dataFile = SD.open(logName, FILE_WRITE);
		
		// se tutto ok lo richiude e basta
		if(dataFile)
			dataFile.close();
		// altrimenti disattiva il log
		else
			sdOK = false;
	}
	
	// scritta LOG spenta
	blink = false;
	
	// inizializza la variabile tim per l'esecuzione ogni secondo
	// del contenuto del loop
	tim = millis();
	
	// forza la lettura della temperatura al primo loop
	// mettendo tempTicks ad un valore grande
	tempTicks = 1000;
	
	// forza l'esecuzione di un log all'avvio
	// mettendo logTicks ad un valore grande
	logTicks = 1000;
}

void loop(void)
{
	// se dall'ultimo loop non è ancora passato almeno un secondo
	// non fa nulla!
	if(millis() < tim)
		return;
	// aggiorna il prossimo istante di lettura
	tim = millis() + 1000;

	// 16 caratteri:    "0123456789ABCDEF"
	char oraTempBuf[] = "00:00:00  33.0°C";
	char dataBuf[]    = "01 Gen 2016     ";
	const char *logMsg;
	if(sdOK)
		logMsg = " LOG";
	else
		logMsg = "NOSD";
	
	// legge ora e data dall'RTC
	DateTime now = rtc.now();
	
	// legge la temperatura dal sensore ogni 10 secondi visto che
	// è un'operazione che richiede tempo
	if(tempTicks >= 10)
	{
		tempSensors.requestTemperatures();
		temp = tempSensors.getTempCByIndex(0);
		tempTicks = 0;
	}
	else
		tempTicks++;
	
	// la funzione sprintf di arduino non supporta la stampa di valori
	// in virgola mobile; usiamo quindi un trucchetto scomponendo la
	// temperatura in 2 numeri interi, prendendo solo un decimale
	int iTemp = (int)temp;
	int fTemp = (int)((temp - iTemp) * 10);
	
	// prepara due righe di testo; la prima con l'ora e la temperatura...
	sprintf(oraTempBuf, "%02d:%02d:%02d  %2d.%01d C",
		now.hour(),
		now.minute(),
		now.second(),
		iTemp, fTemp
	);
	
	// ... e la seconda con la data, inserendo anche una scritta lampeggiante
	// se stiamo facendo il log
	sprintf(dataBuf, "%02d %3s %04d %s",
		now.day(),
		nomiMesi[now.month() - 1],
		now.year(),
		blink ? logMsg : ""

	);

	// cancella il display
	LCD.clear();
	
	// scrive l'ora nella prima riga del display
	LCD.print(oraTempBuf);
	
	// si sposta sulla seconda riga del display
	// e scrive la data
	LCD.setCursor(0, 1);
	LCD.print(dataBuf);
	
	// fa lampeggiare la scritta LOG
	blink = !blink;
	
	// controlla se è ora di fare un log, sempre che la SD sia a posto
	if(sdOK && logTicks >= logInterval)
	{
		File dataFile = SD.open(logName, FILE_WRITE);
		if(dataFile)
		{
			dataFile.println(oraTempBuf);
			dataFile.close();
		}
		logTicks = 0;
	}
	else
		logTicks++;
}
