/*
   (makina.it) 2014, 2017

   @author: Edoardo Mecchina
   @date: 18 Giugno 2017
   @version: v03
       added "\n" to DEBUG_PRINT
       added Enable/Disable of Rct Clock Hw Output
*/

#define DEBUG_LEVEL_INFO
#include <FishinoDebug.h>

#include <Fishino32RTC.h>

//
// Global Variables
//
date_time_t rtcTime;
date_time_t rtcAlarm;
uint32_t  rtcEpochTime;
uint32_t  rtcEpochAlarm;
uint32_t  rtcUnixTime;
uint32_t  rtcUnixAlarm;

bool myAlarmState = false;
const int LEDALARM = 26;

// ------------------------------------------------------
// RTC Interrupt Handler
// ------------------------------------------------------
void rtcIsr(const int &led)
{
	// Here we could directly set the alarm LED.
	//  digitalWrite(led, ON);
	DEBUG_PRINT("Alarm Triggered\n");
	myAlarmState = true;
}

// ------------------------------------------------------
// RTC Show Date Tool
// ------------------------------------------------------
void showDate(const char* txt, const date_time_t *dt)
{
	Serial.print(txt);
	Serial.print(' ');
	Serial.print(2000 + dt->year, DEC);
	Serial.print('/');
	Serial.print(dt->month, DEC);
	Serial.print('/');
	Serial.print(dt->day, DEC);
	Serial.print(' ');
	Serial.print(dt->hour, DEC);
	Serial.print(':');
	Serial.print(dt->minute, DEC);
	Serial.print(':');
	Serial.print(dt->second, DEC);
	Serial.println();
}

///////////////////////////////////////////////////////////
// DEBUG support Code
///////////////////////////////////////////////////////////
#define MSXLOOP 100
const int LEDBLNK1 = 25;
const int LEDBLNK2 = 26;

const char *wdayName[8] = {"Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday", "Invalid"};
const char *amaskName[16] = {"Every half-second", "Every second", "Every 10 seconds", "Every minute",
		"Every 10 minutes", "Every hour", "Once a day", "Once a week",
		"Once a month", "Once a year", "Reserved10", "Reserved11",
		"Reserved12", "Reserved13", "Reserved14", "Reserved15"
							};    // (except when configured for February 29, once every four years)

////////////////////////////////////////////////////////////////////////////////
// RtcPic32Class Register Dump
////////////////////////////////////////////////////////////////////////////////
void debugRtcRegs(const char* txt)
{
	struct rtccon_s rtccon;
	struct rtcalrm_s rtcalrm;
	struct rtctime_s rtctime;
	struct rtcdate_s rtcdate;
	struct alrmtime_s alrmtime;
	struct alrmdate_s alrmdate;

	rtccon.val   = Rtc32._rtccon->val;
	rtcalrm.val  = Rtc32._rtcalrm->val;
	rtctime.val  = Rtc32._rtctime->val;
	rtcdate.val  = Rtc32._rtcdate->val;
	alrmtime.val = Rtc32._alrmtime->val;
	alrmdate.val = Rtc32._alrmdate->val;

	DEBUG_PRINT("%s\n", txt);
	DEBUG_PRINT("OSCCON  : %08X\n", OSCCON);
	DEBUG_PRINT("RTCCON  : %0" PRIX32 "\n", rtccon.val);
	DEBUG_PRINT("      oe: %08X (%s)\n", rtccon.rtcoe, (rtccon.rtcoe == 0) ? "Clock Output Disabled" : "Clock presented on I/O");
	DEBUG_PRINT(" halfsec: %08X\n", rtccon.halfsec);
	DEBUG_PRINT("    sync: %08X\n", rtccon.rtcsync);
	DEBUG_PRINT("     wen: %08X (Write %s)\n", rtccon.rtcwen, (rtccon.rtcwen == 0) ? "Disabled" : "Enabled");
	DEBUG_PRINT("   clkon: %08X (%s%s)\n", rtccon.rtcclkon, "RTCC Clock is ", (rtccon.rtcclkon == 0) ? "not running" : "actively running");
	DEBUG_PRINT("rtsecsel: %08X (%s on RTCC pin)\n", rtccon.rtsecsel, (rtccon.rtsecsel == 0) ? "Seconds Clock" : "Alarm pulse");
	DEBUG_PRINT("    sidl: %08X (%s in Idle mode)\n", rtccon.sidl, (rtccon.sidl == 0) ? "Continue" : "Stops");
	DEBUG_PRINT("      on: %08X (%s)\n", rtccon.on, (rtccon.on == 0) ? "Rtc Stopped" : "Rtc Running");
	DEBUG_PRINT("     cal: %08X\n", rtccon.cal);

	DEBUG_PRINT("RTCALRM : %0" PRIX32 "\n", rtcalrm.val);
	DEBUG_PRINT("    arpt: %8X (Alarm will trigger %d time(s))\n", rtcalrm.arpt, (rtcalrm.arpt + 1));
	DEBUG_PRINT("   amask: %8X (%s)\n", rtcalrm.amask, amaskName[rtcalrm.amask]);
	DEBUG_PRINT(" alrmsyn: %8X\n", rtcalrm.alrmsync);
	DEBUG_PRINT("     piv: %8X (Alarm Pulse Initial Value bit)\n", rtcalrm.piv);
	DEBUG_PRINT("   chime: %8X (Chime %s)\n", rtcalrm.chime, (rtcalrm.chime == 0) ? "Disabled" : "Enabled, ARPT<7:0> rolls from 0x00 to 0xFF");
	DEBUG_PRINT("  alrmen: %8X (%s)\n", rtcalrm.alrmen, (rtcalrm.alrmen == 0) ? "Disabled" : "Enabled");

	DEBUG_PRINT("RTC TIMEDATE  : %0" PRIX32 "  %0" PRIX32 "\n", rtctime.val, rtcdate.val);
	DEBUG_PRINT("      Rtc Time: %02X:%02X:%02X  %04X/%02X/%02X (%s)\n", rtctime.hour, rtctime.min, rtctime.sec, 0X2000 + rtcdate.year, rtcdate.month, rtcdate.day, wdayName[rtcdate.wday]);

	DEBUG_PRINT("ALRM TIMEDATE : %0" PRIX32 "  %0" PRIX32 "\n", alrmtime.val, alrmdate.val);
	alrmdate.year = rtcdate.year;
	DEBUG_PRINT("         Alarm: %02X:%02X:%02X       %02X/%02X\n", alrmtime.hour, alrmtime.min, alrmtime.sec, alrmdate.month, alrmdate.day);
}

// ------------------------------------------------------
// RTC DEBUG Show Tools
// ------------------------------------------------------
void debugShowDate(const char* txt, const date_time_t *dt)
{
	DEBUG_PRINT("%s %02d:%02d:%02d  %04d/%02d/%02d (%s)\n", txt, dt->hour, dt->minute, dt->second, 2000 + dt->year, dt->month, dt->day, wdayName[dt->wday]);
}
void debugShowAlarm(const char* txt, const date_time_t *dt)
{
	DEBUG_PRINT("%s %02d:%02d:%02d       %02d/%02d\n", txt, dt->hour, dt->minute, dt->second, dt->month, dt->day);
}

///////////////////////////////////////////////////////////
// Alternate Orange-Blue Blink until USB serial is opened.
// pSec == 0 means Forever
///////////////////////////////////////////////////////////
void blinkUntilReady(int pSec)
{
	bool ledState = false;
	int loops;

	loops = pSec * (1000 / MSXLOOP);
	pinMode(LEDBLNK1, OUTPUT);
	pinMode(LEDBLNK2, OUTPUT);
	digitalWrite(LEDBLNK1, HIGH);
	digitalWrite(LEDBLNK2, LOW);

	while (!Serial)
	{
		delay(MSXLOOP);
		digitalWrite(LEDBLNK1, ledState);
		ledState = !ledState;
		digitalWrite(LEDBLNK2, ledState);
		if (loops != 0)
			if (--loops == 0)
				break;
	}
	digitalWrite(LEDBLNK1, LOW);
	digitalWrite(LEDBLNK2, LOW);
	pinMode(LEDBLNK1, INPUT);
	pinMode(LEDBLNK2, INPUT);
}

// ------------------------------------------------------
// Setup
// ------------------------------------------------------
void setup()
{
	// Initialize Serial Communication
	// Open serial communications and wait for port to open
	Serial.begin(115200);
	while(!Serial)
		;
	
	DEBUG_PRINT("Debug Initialized\n");

	// Alternate Orange-Blue Blink until USB serial is opened or N seconds.
	blinkUntilReady(30);
	DEBUG_PRINT("Main Port Initialized\n");

	DEBUG_PRINT("Testing Fishino32 RTC\n");

	if (!Rtc32.isRtcRunning())
	{
		DEBUG_PRINT("RTC was not running!\n");
		Rtc32.setupRtc(RTC32_OUT_NONE);
	}
	Rtc32.getRtc(&rtcTime);
	debugShowDate("RTC running :", &rtcTime);
	DEBUG_PRINT("Epoch time: %d\n", date_time_to_epoch(&rtcTime));
	DEBUG_PRINT("Unix time: %d\n", date_time_to_unix(&rtcTime));
	//
	// date_time_set(&rtcTime, 16, 12, 31, 23, 58, 45, -1, NODST);
	date_time_set(&rtcTime, 17, 3, 26, 1, 59, 43, -1, NODST);         // Test inizio DST 2017
	// date_time_set(&rtcTime, 17, 10, 29, 2, 59, 43, -1, DST1);         // Test fine DST 2017
	// date_time_set(&rtcTime, 20, 3, 29, 1, 59, 43, -1, NODST);         // Test inizio DST 2020
	// date_time_set(&rtcTime, 20, 10, 25, 2, 59, 43, -1, DST1);         // Test fine DST 2020

	Rtc32.setRtc(&rtcTime);
	Rtc32.getRtc(&rtcTime);
	showDate("Setting Time  to ", &rtcTime);
	debugShowDate("Time set to  :", &rtcTime);

	pinMode(LEDALARM, OUTPUT);
	DEBUG_PRINT("RTC Module Setup Done\n");
	//
	// Setting first ALARM some seconds later
	//
	rtcEpochTime = Rtc32.getRtc(&rtcTime);
	rtcEpochAlarm = rtcEpochTime + 23;
	epoch_to_date_time(&rtcAlarm, rtcEpochAlarm);
	Rtc32.setAlarm(&rtcAlarm, RTC32_AMASK_1DAY, true);
	Rtc32.attachInterrupt(rtcIsr, LEDALARM);
	//
	Rtc32.getAlarm(&rtcAlarm);
	showDate("ALARM set to", &rtcAlarm);
	debugShowAlarm("ALARM set to:", &rtcAlarm);

	debugRtcRegs("Initial Register Settings");
}
// ------------------------------------------------------
// Loop
// ------------------------------------------------------
void loop()
{
	uint32_t rtcEpochNow;
	// put your main code here, to run repeatedly:
	rtcEpochNow = Rtc32.getRtc(&rtcTime);
	if ((rtcEpochNow - rtcEpochTime) >= 5)
	{
		rtcEpochTime = rtcEpochNow;
		showDate("\nTIME is ", &rtcTime);
		debugShowDate("TIME is      :", &rtcTime);
		debugRtcRegs("Actual Register Settings");
	}
	if (myAlarmState)
	{
		digitalWrite(LEDALARM, HIGH);
		rtcEpochTime = Rtc32.getRtc(&rtcTime);
		showDate("ALARM event at", &rtcTime);
		debugShowDate(" Alarm Event :", &rtcTime);
		//
		delay(500);
		digitalWrite(LEDALARM, LOW);
		//
		rtcEpochAlarm = rtcEpochTime + 23;
		epoch_to_date_time(&rtcAlarm, rtcEpochAlarm);
		Rtc32.setAlarm(&rtcAlarm, RTC32_AMASK_1DAY, true);
		Rtc32.attachInterrupt(rtcIsr, LEDALARM);
		//
		Rtc32.getAlarm(&rtcAlarm);
		showDate("ALARM set to  ", &rtcAlarm);
		debugShowAlarm("ALARM set to:", &rtcAlarm);
		//
		myAlarmState = false;
	}
}
