// 
// Released to the public domain! Enjoy!
// v02 - 26/04/2017 
// v03 - 18/06/2017
// for Fishino32 by Edoardo Mecchina
//

#ifndef _RTCPIC32_H_
#define _RTCPIC32_H_

#include <Arduino.h>

// TimeZone Correction
#define TIMEZONE_SECONDS(TZ)	(3600 * TZ)

// DST Correction
#define NODST	0
#define DST1	1

// Accessing CLR - SET - INV registers
#define	CLROFFSET(A)	(A+1)
#define	SETOFFSET(A)	(A+2)
#define	INVOFFSET(A)	(A+3)

// Alarm MASK modes
#define RTC32_AMASK_HALFSEC 0x00000000
#define RTC32_AMASK_1SEC	(RTC32_AMASK_HALFSEC + 0x0100)
#define RTC32_AMASK_10SEC 	(RTC32_AMASK_1SEC + 0x0100)
#define RTC32_AMASK_1MIN	(RTC32_AMASK_10SEC + 0x0100)
#define RTC32_AMASK_10MIN	(RTC32_AMASK_1MIN + 0x0100)
#define RTC32_AMASK_1HOUR	(RTC32_AMASK_10MIN + 0x0100)
#define RTC32_AMASK_1DAY	(RTC32_AMASK_1HOUR + 0x0100)
#define RTC32_AMASK_1WEEK	(RTC32_AMASK_1DAY + 0x0100)
#define RTC32_AMASK_1MONTH	(RTC32_AMASK_1WEEK + 0x0100)
#define RTC32_AMASK_1YEAR	(RTC32_AMASK_1MONTH + 0x0100)

// Alarm OUTPUT mode
#define RTC32_OUT_NONE		0X00000000
#define RTC32_OUT_ALARM		0X00000001
#define RTC32_OUT_CLOCK		0X00000081
#define RTC32_OUT_MASK		0X00000081

typedef void (*RTC32_INTERRUPT_HANDLER)(void *param);

// PIC32MX - RTCC Registers structures
// Section 29. Real-Time Clock and Calendar (RTCC)
//
// Register 29-1: RTCCON: RTC Control Register
struct rtccon_s
{
	union
	{
		uint32_t val;
		struct
		{
			unsigned rtcoe:1;
			unsigned halfsec:1;
			unsigned rtcsync:1;
			unsigned rtcwen:1;
			unsigned _space1:2;
			unsigned rtcclkon:1;
			unsigned rtsecsel:1;
			unsigned _space2:5;
			unsigned sidl:1;
			unsigned _space3:1;
			unsigned on:1;
			unsigned cal:10;
			unsigned _space4:6;
		};
	};
} __attribute__((packed));
//
// Register 29-2: RTCALRM: RTC ALARM Control Register
struct rtcalrm_s
{
	union
	{
		uint32_t val;
		struct
		{
			unsigned arpt:8;
			unsigned amask:4;
			unsigned alrmsync:1;
			unsigned piv:1;
			unsigned chime:1;
			unsigned alrmen:1;
			unsigned _space1:16;
		};
	};
} __attribute__((packed));
//
// Register 29-3: RTCTIME: RTC Time Value Register
struct rtctime_s
{
	union
	{
		uint32_t val;
		struct
		{
			unsigned _null1:8;
      unsigned sec:7;
			unsigned _secnull:1;
      unsigned min:7;
			unsigned _minnull:1;
      unsigned hour:6;
			unsigned _hournull:2;
		};
	};
} __attribute__((packed));
//
// Register 29-4: RTCDATE: RTC Date Value Register
struct rtcdate_s
{
	union
	{
		uint32_t val;
		struct
		{
			unsigned wday:3;
			unsigned _wdaynull:5;
      unsigned day:6;
			unsigned _daynull:2;
      unsigned month:5;
			unsigned _monthnull:3;
      unsigned year:8;
		};
	};
} __attribute__((packed));
//
// Register 29-5: ALRMTIME: Alarm Time Value Register
struct alrmtime_s
{
	union
	{
		uint32_t val;
		struct
		{
			unsigned _null1:8;
      unsigned sec:7;
			unsigned _secnull:1;
      unsigned min:7;
			unsigned _minnull:1;
    	unsigned hour:6;
			unsigned _hournull:2;
		};
	};
} __attribute__((packed));
//
// Register 29-6: ALRMDATE: Alarm Date Value Register
struct alrmdate_s
{
	union
	{
		uint32_t val;
		struct
		{
			unsigned wday:3;
			unsigned _wdaynull:5;
      unsigned day:6;
			unsigned _daynull:2;
      unsigned month:5;
			unsigned _monthnull:3;
      unsigned year:8;
 		};
	};
} __attribute__((packed));

//
struct date_time_t
{
  unsigned char second; // 0-59
  unsigned char minute; // 0-59
  unsigned char hour;   // 0-23
  unsigned char wday;   // 0-6
  unsigned char day;    // 1-31
  unsigned char month;  // 1-12
  unsigned char year;   // 0-99 (representing 2000-2099)
};

//
unsigned int date_time_set(date_time_t *dt, const int year, const int month, const int day, const int hour, const int minute, const int second, const int wday, const int dst);
unsigned int date_time_to_epoch(date_time_t *dt);
void epoch_to_date_time(date_time_t *date_time, unsigned int epoch);
unsigned int date_time_to_unix(date_time_t *dt);
unsigned int day_of_week(const int year, const int month, const int day, const int hour, const int minute, const int second);
int date_time_is_dst(date_time_t *dt);
bool epoch_is_dst(unsigned int epoch);
//
//
// RTC based on the PIC32 Internal RTC
// 
class RtcPic32Class
{
	friend void _RTC32_Initialize(RtcPic32Class &t, uint8_t index);
	friend void _rtcInterruptHandler(void);

	private:
		int _vec;
		int _irq;
		int _spl;
		int _ipl;

		// external interrupt handler pointer
		// used to set PIC interrupt vector
		void (*_isrHandler)(void);
		
		// internal interrupt handler
		void isr(void);
		
		// user interrupt handler -- he can pass a parameter
		// stored previously by attachInterrupt() call
		void *_handlerParameter;
		RTC32_INTERRUPT_HANDLER _rtcInterruptHandler;
		
		// flag for enabled interrupts
		bool _interruptsEnabled;
		
		// helper for parameter-safe attachinterrupt
		void _attachInterrupt(RTC32_INTERRUPT_HANDLER handler, void *param);

	public:
		volatile rtccon_s *_rtccon;
		volatile rtcalrm_s *_rtcalrm;
		volatile rtctime_s *_rtctime;
		volatile rtcdate_s *_rtcdate;
		volatile alrmtime_s *_alrmtime;
		volatile alrmdate_s *_alrmdate;
		int GMT;
		bool DST;
		// constructor
		RtcPic32Class();
		
		// destructor
		~RtcPic32Class();

		uint32_t isRtcRunning(void);
		uint32_t isAlarmRunning(void);
		void setupRtc(uint32_t mode);
		void setRtc(date_time_t *dt);
		void setAlarm(date_time_t *dt, uint32_t mode, bool enable);

		void resetAlarm(void);
		uint32_t getRtc(date_time_t *dt);
		uint32_t getAlarm(date_time_t *dt);

				// attach an interrupt handler to timer passing a parameter
		template<class T>void attachInterrupt(void (*isr)(T &param), T &param)
			{ _attachInterrupt((RTC32_INTERRUPT_HANDLER)isr, (void *)&param); }

		// attach an interrupt handler to timer with no parameters
		void attachInterrupt(void (*isr)(void))
			{ _attachInterrupt((RTC32_INTERRUPT_HANDLER)isr, NULL); }

		// detach the interrupt handler from timer
		void detachInterrupt();
		
		// set interrupt priority
		void setInterruptPriority(int pri);		

		// enable interrupts
		void enInterrupts(void);
		
		// disable interrupts
		void disInterrupts(void);
						
};

extern RtcPic32Class &_rtc1(void);
#define Rtc32 _rtc1()

#endif // _RTCPIC32_H_
