/////////////////////////////////////////////////////////////////////////////////
//
//  Fishino Recorder
//	A class to record audio samples on an SD card
//  Available formata :
//		.wav (PCM, uncompressed), full support
//		.mp3, limited support, usable only on low rate audio streams
//
//  Copyright(C) 2016 by Massimo Del Fedele. All right reserved.
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
/////////////////////////////////////////////////////////////////////////////////

#ifndef _FISHINORECORDER_H
#define _FISHINORECORDER_H

// for sending audio to codec
#include <I2S.h>

// for codec interface
#include <FishinoAudioCodec.h>

// for audio writer classes
#include <FishinoAudioWriter.h>

class FishinoRecorder
{
	friend void i2sHandler(FishinoRecorder &r);

	private:
	
		// the audio writer for this file
		AudioWriter *_writer;
		
		// the I2S interface
		I2SClass *_i2s;
		
		// the used codec
		AudioCodec *_codec;
		
		// handle i2s requests
		void handleI2SRequest(void);
		
		// play in progress flag
		volatile bool _recording;
		
		// we need to store audioformat here
		// because it will be set on writer creation
		uint8_t _bits;
		uint32_t _sampleRate;
		ChannelModes _channelMode;
		
		// some info grabbed from writer before closing

		// count of skipped samples
		uint32_t _curSkippedSamples;
		uint32_t _totalSkippedSamples;
		uint32_t _maxSkippedSamples;
		
		// total recorded samples
		uint32_t _numSamples;
		
		// debugger channel
		Stream *_debugStream;

	protected:
	
	public:
	
		// constructor
		FishinoRecorder(I2SClass &i2s = I2S, AudioCodec &codec = ALC5631);
	
		// destructor
		~FishinoRecorder();
		
		// set the audio format for recording
		bool setAudioFormat(uint32_t sampleRate, uint8_t bits, ChannelModes channelMode);
		
		// set compression
		bool setCompression(uint8_t mode);
		
		// open the record file and prepare for recording
		// but don't start it
		bool open(const char *fileName);
		
		// close and finish recording
		bool close(void);
	
		// start/resume recording a file to SD card
		// if fileName is not given, just use previous one
		// or restart record if paused
		bool record(const char *fileName = NULL);
		
		// pause the recording
		bool pause(void);
		
		// stop recording and close file on SD card
		bool stop(void);
		
		// check if it's still recording
		bool isRecording(void) { return _recording; }
		
		// change the input gain
		bool setMicGain(float vol, float bal = 0);
		
		// change the input volume
		bool setInputVolume(float vol, float bal = 0);
		
		// get some info from writer
		uint32_t getNumSamples(void);
		uint32_t getMaxSkippedSamples(void);
		uint32_t getTotalSkippedSamples(void);

		// enable/disable debug logs
		void debug(Stream &s = Serial);
		void noDebug(void);
};

#endif