//////////////////////////////////////////////////////////////////////////////////////
//						 		FishinoDebug.cpp									//
//																					//
//						Some debug helpers for Fishino32							//
//																					//
//		Copyright (c) 2017 Massimo Del Fedele.  All rights reserved.				//
//																					//
//	Redistribution and use in source and binary forms, with or without				//
//	modification, are permitted provided that the following conditions are met:		//
//																					//
//	- Redistributions of source code must retain the above copyright notice,		//
//	  this list of conditions and the following disclaimer.							//
//	- Redistributions in binary form must reproduce the above copyright notice,		//
//	  this list of conditions and the following disclaimer in the documentation		//
//	  and/or other materials provided with the distribution.						//
//																					//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"		//
//	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE		//
//	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE		//
//	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE		//
//	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR				//
//	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF			//
//	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS		//
//	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN			//
//	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)			//
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE		//
//	POSSIBILITY OF SUCH DAMAGE.														//
//																					//
//  Version 5.1.0 -- April 2017		Initial version									//
//  Version 5.1.1 -- May 2017		Removed \n on DEBUG_PRINT macro					//
//  Version 5.2.0 -- May 2017		Renamed in FishinoDebug.h						//
//									Uniformed 8 and 32 bit boards code				//
//  Version 6.0.0 -- June 2017		Added stack dump code for 32 bit boards			//
//  Version 6.0.4 -- June 2017		Added memory info functions						//
//  Version 7.1.0 - 20/10/2017		Better free memory display						//
//																					//
//////////////////////////////////////////////////////////////////////////////////////
#include "FishinoDebug.h"

#ifndef ARDUINO_STREAMING
#define ARDUINO_STREAMING
template<class T> inline Print &operator <<(Print &stream, T arg)
{
	stream.print(arg);
	return stream;
}
#endif

#ifndef _FISHINO_PIC32_
#include "avr/pgmspace.h"
#endif

Print *__debug_serial = &Serial;

extern "C"
{
	int __debug_putchar (char c, FILE *stream)
	{
	    __debug_serial->write(c);
	    return 0 ;
	}
	
#ifndef _FISHINO_PIC32_
	int __debug_vprintf_P(PGM_P formatP, va_list ap)
	{
		int ret;
		
		size_t fmtLen = strlen_P(formatP);
		char* format = new char[fmtLen + 1];
		strcpy_P(format, formatP);
		ret = vprintf(format, ap);
		delete[] format;
		return ret;
	}
#else
	struct _reent;
	long _write_r(struct _reent *ptr, int fd, const void *buf, size_t cnt)
	{
		size_t l = cnt;
		const char *p = (const char *)buf;
		while(l--)
			__debug_serial->write(*p++);
	    return cnt;
	}
#endif

	void __debug__printf__(const __FlashStringHelper *fmt, ...)
	{
		static bool initialized = false;
		if(!initialized)
		{
#ifndef _FISHINO_PIC32_
			static FILE __debug_file = {0};
			fdev_setup_stream (&__debug_file, __debug_putchar, NULL, _FDEV_SETUP_WRITE);
			stdout = &__debug_file;
#else
			setbuf(stdin, NULL);
			setbuf(stdout, NULL);
#endif
			initialized = true;
		}

		// resulting string limited to 127 chars
		// avoid allocating statically on 8 bit boards to spare ram
		va_list args;
		va_start(args, (const char *)fmt);
#ifndef _FISHINO_PIC32_
		if(__debug_vprintf_P((const char *)fmt, args) < 0)
#else
		if(vprintf((const char *)fmt, args) < 0)
#endif
			/*__debug_serial->print("*ERR*\n")*/;
		va_end(args);
	}
	
	// hex dump
	#define HEXDUMP_ROW_SIZE 16
	#define BINDUMP_ROW_SIZE 16
	
	static char __nibble2char(uint8_t nibble)
	{
		nibble &= 0x0f;
		if(nibble < 0x0a)
			return nibble + '0';
		else
			return nibble + 'a' - 10;
	}
	
	void __debug__hexDump(const void *ptr, uint32_t len)
	{
		const uint8_t *bPtr = (const uint8_t *)ptr;
		uint32_t rows = (len + HEXDUMP_ROW_SIZE - 1) / HEXDUMP_ROW_SIZE;
		for(uint32_t row = 0; row < rows && len; row++)
		{
			for(uint32_t col = 0; col < HEXDUMP_ROW_SIZE && len; col++, len--)
			{
				uint8_t b = *bPtr++;
				__debug_serial->print(__nibble2char(b >> 4));
				__debug_serial->print(__nibble2char(b));
				__debug_serial->print(' ');
			}
			__debug_serial->print("\r\n");
		}
	}
	
	// binary dump
	void __debug__binDump(void const *ptr, uint32_t len)
	{
		const uint8_t *bPtr = (const uint8_t *)ptr;
		uint32_t rows = (len + BINDUMP_ROW_SIZE - 1) / BINDUMP_ROW_SIZE;
		for(uint32_t row = 0; row < rows && len; row++)
		{
			for(uint32_t col = 0; col < BINDUMP_ROW_SIZE && len; col++, len--)
			{
				uint8_t b = *bPtr++;
				uint8_t mask = 0x80;
				for(uint8_t i = 0; i < 8; i++)
				{
					__debug_serial->print( (b & mask) ? "1" : "0");
					mask >>= 1;
				}
				__debug_serial->print(' ');
			}
			__debug_serial->print("\r\n");
		}
	}

	
	void __debug__set__stream(void *s)
	{
		__debug_serial = (Print *)s;
	}
}
