//////////////////////////////////////////////////////////////////////////////////////
//	This is our library for the Adafruit ILI9341 Breakout and Shield				//
//	----> http://www.adafruit.com/products/1651										//
//																					//	
//	Check out the links above for our tutorials and wiring diagrams					//
//	These displays use SPI to communicate, 4 or 5 pins are required to				//
//	interface (RST is optional)														//
//	Adafruit invests time and resources providing this open source code,			//
//	please support Adafruit and open-source hardware by purchasing					//
//	products from Adafruit!															//
//																					//	
//	Written by Limor Fried/Ladyada for Adafruit Industries.							//
//	MIT license, all text above must be included in any redistribution				//
//																					//	
//	Modified on April, 2017 by Massimo Del Fedele for Fishino boards and TFT Shield	//
//////////////////////////////////////////////////////////////////////////////////////
#include "FishinoILI9341.h"
#ifdef __AVR
	#include <avr/pgmspace.h>
#endif

#include "pins_arduino.h"
#include "wiring_private.h"
#
#include <limits.h>
#include <SPI.h>


// If the SPI library has transaction support, these functions
// establish settings and protect from interference from other
// libraries.  Otherwise, they simply do nothing.
static inline void spi_begin(void) __attribute__((always_inline));
static inline void spi_begin(void)
{
#if defined(_FISHINO32_) || defined(_FISHINO32_96M_)
	SPI.beginTransaction(SPISettings(12000000, MSBFIRST, SPI_MODE0));
#elif defined(_FISHINO_PIRANHA_) || defined(_FISHINO_PIRANHA_96M_)
	SPI.beginTransaction(SPISettings(16000000, MSBFIRST, SPI_MODE0));
#else
	SPI.beginTransaction(SPISettings(24000000, MSBFIRST, SPI_MODE0));
#endif
}

static inline void spi_end(void) __attribute__((always_inline));
static inline void spi_end(void)
{
	SPI.endTransaction();
}

// Constructor when using hardware SPI.  Faster, but must use SPI pins
// specific to each board type (e.g. 11,13 for Uno, 51,52 for Mega, etc.)
FishinoILI9341Class::FishinoILI9341Class() : FishinoGFX(ILI9341_TFTWIDTH, ILI9341_TFTHEIGHT)
{
}

void FishinoILI9341Class::spiwrite(uint8_t c)
{
#if defined (__AVR__)
	SPDR = c;
	while(!(SPSR & _BV(SPIF)))
		;
#else
	SPI.transfer(c);
#endif
}

void FishinoILI9341Class::writecommand(uint8_t c)
{
#if defined (USE_FAST_PINIO)
	*dcport &= ~dcpinmask;
	*csport &= ~cspinmask;
#else
	digitalWrite(_dc, LOW);
	digitalWrite(_cs, LOW);
#endif

	spiwrite(c);

#if defined (USE_FAST_PINIO)
	*csport |= cspinmask;
#else
	digitalWrite(_cs, HIGH);
#endif
}

void FishinoILI9341Class::writedata(uint8_t c)
{
#if defined (USE_FAST_PINIO)
	*dcport |=  dcpinmask;
	*csport &= ~cspinmask;
#else
	digitalWrite(_dc, HIGH);
	digitalWrite(_cs, LOW);
#endif

	spiwrite(c);

#if defined (USE_FAST_PINIO)
	*csport |= cspinmask;
#else
	digitalWrite(_cs, HIGH);
#endif
}

// Rather than a bazillion writecommand() and writedata() calls, screen
// initialization commands and arguments are organized in these tables
// stored in PROGMEM.  The table may look bulky, but that's mostly the
// formatting -- storage-wise this is hundreds of bytes more compact
// than the equivalent code.  Companion function follows.
#define DELAY 0x80

// Companion code to the above tables.  Reads and issues
// a series of LCD commands stored in PROGMEM byte array.
void FishinoILI9341Class::commandList(uint8_t *addr)
{

	uint8_t  numCommands, numArgs;
	uint16_t ms;

	numCommands = pgm_read_byte(addr++);   // Number of commands to follow
	while (numCommands--)                  // For each command...
	{
		writecommand(pgm_read_byte(addr++)); //   Read, issue command
		numArgs  = pgm_read_byte(addr++);    //   Number of args to follow
		ms       = numArgs & DELAY;          //   If hibit set, delay follows args
		numArgs &= ~DELAY;                   //   Mask out delay bit
		while (numArgs--)                    //   For each argument...
		{
			writedata(pgm_read_byte(addr++));  //     Read, issue argument
		}

		if (ms)
		{
			ms = pgm_read_byte(addr++); // Read post-command delay time (ms)
			if (ms == 255) ms = 500;    // If 255, delay for 500 ms
			delay(ms);
		}
	}
}

void FishinoILI9341Class::begin(uint8_t cs, uint8_t dc)
{
	_cs   = cs;
	_dc   = dc;
	pinMode(_dc, OUTPUT);
	pinMode(_cs, OUTPUT);

#if defined (USE_FAST_PINIO)
	csport    = portOutputRegister(digitalPinToPort(_cs));
	cspinmask = digitalPinToBitMask(_cs);
	dcport    = portOutputRegister(digitalPinToPort(_dc));
	dcpinmask = digitalPinToBitMask(_dc);
#endif

	SPI.begin();

	spi_begin();
	writecommand(0xEF);
	writedata(0x03);
	writedata(0x80);
	writedata(0x02);

	writecommand(0xCF);
	writedata(0x00);
	writedata(0XC1);
	writedata(0X30);

	writecommand(0xED);
	writedata(0x64);
	writedata(0x03);
	writedata(0X12);
	writedata(0X81);

	writecommand(0xE8);
	writedata(0x85);
	writedata(0x00);
	writedata(0x78);

	writecommand(0xCB);
	writedata(0x39);
	writedata(0x2C);
	writedata(0x00);
	writedata(0x34);
	writedata(0x02);

	writecommand(0xF7);
	writedata(0x20);

	writecommand(0xEA);
	writedata(0x00);
	writedata(0x00);

	writecommand(ILI9341_PWCTR1);    //Power control
	writedata(0x23);   //VRH[5:0]

	writecommand(ILI9341_PWCTR2);    //Power control
	writedata(0x10);   //SAP[2:0];BT[3:0]

	writecommand(ILI9341_VMCTR1);    //VCM control
	writedata(0x3e); //ﾶￔﾱ￈ﾶ￈ﾵ￷ﾽￚ
	writedata(0x28);

	writecommand(ILI9341_VMCTR2);    //VCM control2
	writedata(0x86);  //--

	writecommand(ILI9341_MADCTL);    // Memory Access Control
	writedata(0x48);

	writecommand(ILI9341_PIXFMT);
	writedata(0x55);

	writecommand(ILI9341_FRMCTR1);
	writedata(0x00);
	writedata(0x18);

	writecommand(ILI9341_DFUNCTR);    // Display Function Control
	writedata(0x08);
	writedata(0x82);
	writedata(0x27);

	writecommand(0xF2);    // 3Gamma Function Disable
	writedata(0x00);

	writecommand(ILI9341_GAMMASET);    //Gamma curve selected
	writedata(0x01);

	writecommand(ILI9341_GMCTRP1);    //Set Gamma
	writedata(0x0F);
	writedata(0x31);
	writedata(0x2B);
	writedata(0x0C);
	writedata(0x0E);
	writedata(0x08);
	writedata(0x4E);
	writedata(0xF1);
	writedata(0x37);
	writedata(0x07);
	writedata(0x10);
	writedata(0x03);
	writedata(0x0E);
	writedata(0x09);
	writedata(0x00);

	writecommand(ILI9341_GMCTRN1);    //Set Gamma
	writedata(0x00);
	writedata(0x0E);
	writedata(0x14);
	writedata(0x03);
	writedata(0x11);
	writedata(0x07);
	writedata(0x31);
	writedata(0xC1);
	writedata(0x48);
	writedata(0x08);
	writedata(0x0F);
	writedata(0x0C);
	writedata(0x31);
	writedata(0x36);
	writedata(0x0F);

	writecommand(ILI9341_SLPOUT);    //Exit Sleep
	spi_end();
	delay(120);
	spi_begin();
	writecommand(ILI9341_DISPON);    //Display on
	spi_end();
}

void FishinoILI9341Class::setAddrWindow(uint16_t x0, uint16_t y0, uint16_t x1, uint16_t y1)
{
	writecommand(ILI9341_CASET); // Column addr set
	writedata(x0 >> 8);
	writedata(x0 & 0xFF);     // XSTART
	writedata(x1 >> 8);
	writedata(x1 & 0xFF);     // XEND

	writecommand(ILI9341_PASET); // Row addr set
	writedata(y0>>8);
	writedata(y0);     // YSTART
	writedata(y1>>8);
	writedata(y1);     // YEND

	writecommand(ILI9341_RAMWR); // write to RAM
}

void FishinoILI9341Class::pushColor(uint16_t color)
{
	spi_begin();

#if defined(USE_FAST_PINIO)
	*dcport |=  dcpinmask;
	*csport &= ~cspinmask;
#else
	digitalWrite(_dc, HIGH);
	digitalWrite(_cs, LOW);
#endif

	spiwrite(color >> 8);
	spiwrite(color);

#if defined(USE_FAST_PINIO)
	*csport |= cspinmask;
#else
	digitalWrite(_cs, HIGH);
#endif

	spi_end();
}

void FishinoILI9341Class::pushColors(uint16_t nColors, uint16_t const*colors)
{
	spi_begin();

#if defined(USE_FAST_PINIO)
	*dcport |=  dcpinmask;
	*csport &= ~cspinmask;
#else
	digitalWrite(_dc, HIGH);
	digitalWrite(_cs, LOW);
#endif

	for(uint16_t i = 0; i < nColors; i++)
	{
		spiwrite(*colors >> 8);
		spiwrite(*colors++);
	}


#if defined(USE_FAST_PINIO)
	*csport |= cspinmask;
#else
	digitalWrite(_cs, HIGH);
#endif

	spi_end();
}


void FishinoILI9341Class::drawPixel(int16_t x, int16_t y, uint16_t color)
{
	if ((x < 0) ||(x >= _width) || (y < 0) || (y >= _height))
		return;

	spi_begin();
	setAddrWindow(x,y,x+1,y+1);

#if defined(USE_FAST_PINIO)
	*dcport |=  dcpinmask;
	*csport &= ~cspinmask;
#else
	digitalWrite(_dc, HIGH);
	digitalWrite(_cs, LOW);
#endif

	spiwrite(color >> 8);
	spiwrite(color);

#if defined(USE_FAST_PINIO)
	*csport |= cspinmask;
#else
	digitalWrite(_cs, HIGH);
#endif

	spi_end();
}

void FishinoILI9341Class::drawFastVLine(int16_t x, int16_t y, int16_t h, uint16_t color)
{
	// Rudimentary clipping
	if ((x >= _width) || (y >= _height))
		return;

	if ((y+h-1) >= _height)
		h = _height-y;

	spi_begin();
	setAddrWindow(x, y, x, y+h-1);

	uint8_t hi = color >> 8, lo = color;

#if defined(USE_FAST_PINIO)
	*dcport |=  dcpinmask;
	*csport &= ~cspinmask;
#else
	digitalWrite(_dc, HIGH);
	digitalWrite(_cs, LOW);
#endif

	while (h--)
	{
		spiwrite(hi);
		spiwrite(lo);
	}

#if defined(USE_FAST_PINIO)
	*csport |= cspinmask;
#else
	digitalWrite(_cs, HIGH);
#endif

	spi_end();
}

void FishinoILI9341Class::drawFastHLine(int16_t x, int16_t y, int16_t w, uint16_t color)
{
	// Rudimentary clipping
	if ((x >= _width) || (y >= _height))
		return;
	if ((x+w-1) >= _width)
		w = _width-x;
	spi_begin();
	setAddrWindow(x, y, x+w-1, y);

	uint8_t hi = color >> 8, lo = color;
#if defined(USE_FAST_PINIO)
	*dcport |=  dcpinmask;
	*csport &= ~cspinmask;
#else
	digitalWrite(_dc, HIGH);
	digitalWrite(_cs, LOW);
#endif
	while (w--)
	{
		spiwrite(hi);
		spiwrite(lo);
	}
#if defined(USE_FAST_PINIO)
	*csport |= cspinmask;
#else
	digitalWrite(_cs, HIGH);
#endif
	spi_end();
}

void FishinoILI9341Class::fillScreen(uint16_t color)
{
	fillRect(0, 0,  _width, _height, color);
}

// fill a rectangle
void FishinoILI9341Class::fillRect(int16_t x, int16_t y, int16_t w, int16_t h, uint16_t color)
{
	// rudimentary clipping (drawChar w/big text requires this)
	if ((x >= _width) || (y >= _height))
		return;
	if ((x + w - 1) >= _width)
		w = _width  - x;
	if ((y + h - 1) >= _height)
		h = _height - y;

	spi_begin();
	setAddrWindow(x, y, x+w-1, y+h-1);

	uint8_t hi = color >> 8, lo = color;

#if defined(USE_FAST_PINIO)
	*dcport |=  dcpinmask;
	*csport &= ~cspinmask;
#else
	digitalWrite(_dc, HIGH);
	digitalWrite(_cs, LOW);
#endif

	for (y=h; y>0; y--)
	{
		for (x=w; x>0; x--)
		{
			spiwrite(hi);
			spiwrite(lo);
		}
	}
#if defined(USE_FAST_PINIO)
	*csport |= cspinmask;
#else
	digitalWrite(_cs, HIGH);
#endif

	spi_end();
}

// Pass 8-bit (each) R,G,B, get back 16-bit packed color
uint16_t FishinoILI9341Class::color565(uint8_t r, uint8_t g, uint8_t b)
{
	return ((r & 0xF8) << 8) | ((g & 0xFC) << 3) | (b >> 3);
}


#define MADCTL_MY  0x80
#define MADCTL_MX  0x40
#define MADCTL_MV  0x20
#define MADCTL_ML  0x10
#define MADCTL_RGB 0x00
#define MADCTL_BGR 0x08
#define MADCTL_MH  0x04

void FishinoILI9341Class::setRotation(uint8_t m)
{

	spi_begin();
	writecommand(ILI9341_MADCTL);
	rotation = m % 4; // can't be higher than 3
	switch (rotation)
	{
		case 0:
			writedata(MADCTL_MX | MADCTL_BGR);
			_width  = ILI9341_TFTWIDTH;
			_height = ILI9341_TFTHEIGHT;
			break;
		case 1:
			writedata(MADCTL_MV | MADCTL_BGR);
			_width  = ILI9341_TFTHEIGHT;
			_height = ILI9341_TFTWIDTH;
			break;
		case 2:
			writedata(MADCTL_MY | MADCTL_BGR);
			_width  = ILI9341_TFTWIDTH;
			_height = ILI9341_TFTHEIGHT;
			break;
		case 3:
			writedata(MADCTL_MX | MADCTL_MY | MADCTL_MV | MADCTL_BGR);
			_width  = ILI9341_TFTHEIGHT;
			_height = ILI9341_TFTWIDTH;
			break;
	}
	spi_end();
}


void FishinoILI9341Class::invertDisplay(boolean i)
{
	spi_begin();
	writecommand(i ? ILI9341_INVON : ILI9341_INVOFF);
	spi_end();
}

FishinoILI9341Class &__GetFishinoILI9341(void)
{
	static FishinoILI9341Class ili;
	return ili;
}
