//////////////////////////////////////////////////////////////////////////////////////
//																					//
//								FishinoTimer32.h									//
//	Library to handle Fishino's timers in uniform way between 8 and 32 bit boards	//
//					Created by Massimo Del Fedele, 2019								//
//																					//
//  Copyright (c) 2019 Massimo Del Fedele.  All rights reserved.					//
//																					//
//	Redistribution and use in source and binary forms, with or without				//
//	modification, are permitted provided that the following conditions are met:		//
//																					//
//	- Redistributions of source code must retain the above copyright notice,		//
//	  this list of conditions and the following disclaimer.							//
//	- Redistributions in binary form must reproduce the above copyright notice,		//
//	  this list of conditions and the following disclaimer in the documentation		//
//	  and/or other materials provided with the distribution.						//
//																					//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"		//
//	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE		//
//	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE		//
//	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE		//
//	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR				//
//	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF			//
//	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS		//
//	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN			//
//	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)			//
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE		//
//	POSSIBILITY OF SUCH DAMAGE.														//
//																					//
//	VERSION 7.6.0 - INITIAL VERSION													//
//																					//
//////////////////////////////////////////////////////////////////////////////////////
typedef void (*FISHINOTIMER_INTERRUPT_HANDLER)(void *param);

enum FishinoTimerSource
{
	FISHINOTIMER_SOURCE_INTERNAL	= 0,
	FISHINOTIMER_SOURCE_EXTERNAL	= 1
};

enum FishinoTimerPrescale
{
	FISHINOTIMER_PRESCALE_1,
	FISHINOTIMER_PRESCALE_2,
	FISHINOTIMER_PRESCALE_4,
	FISHINOTIMER_PRESCALE_8,
	FISHINOTIMER_PRESCALE_16,
	FISHINOTIMER_PRESCALE_32,
	FISHINOTIMER_PRESCALE_64,
	FISHINOTIMER_PRESCALE_256
};

struct tcon_s
{
	union
	{
		uint32_t val;
		struct
		{
			unsigned _unused:1;
			unsigned tcs:1;
			unsigned tsync:1;
			unsigned t32:1;
			unsigned tckps:3;
			unsigned tgate:1;
			unsigned _space:3;
			unsigned twip:1;
			unsigned twdis:1;
			unsigned sidl:1;
			unsigned frz:1;
			unsigned on:1;
			unsigned _blank:16;
		};
	};
} __attribute__((packed));

class FishinoTimerClass
{
	friend void _FishinoTimer_Initialize(FishinoTimerClass &t, uint8_t index);

	private:

		friend void _timer1InterruptHandler(void);
		friend void _timer2InterruptHandler(void);
		friend void _timer3InterruptHandler(void);
		friend void _timer4InterruptHandler(void);
		friend void _timer5InterruptHandler(void);
		friend void _timer23InterruptHandler(void);
		friend void _timer45InterruptHandler(void);

		int _vec;
		int _irq;
		int _spl;
		int _ipl;
		volatile uint32_t *_pr;
		volatile uint32_t *_tmr;
		volatile tcon_s *_tcon;
		int _ps_size;
		
		// external interrupt handler pointer
		// used to set PIC interrupt vector
		void (*_isrHandler)(void);
		
		// internal interrupt handler
		void isr(void);
		
		// user interrupt handler -- he can pass a parameter
		// stored previously by attachInterrupt() call
		void *_handlerParameter;
		FISHINOTIMER_INTERRUPT_HANDLER _interruptHandler;
		
		// flag for enabled interrupts
		bool _interruptsEnabled;
		
		// helper for parameter-safe attachinterrupt
		void _attachInterrupt(FISHINOTIMER_INTERRUPT_HANDLER handler, void *param);

	public:
		// constructor
		FishinoTimerClass();
		
		// destructor
		~FishinoTimerClass();
		
		// set timer frequenency
		bool setFrequency(uint32_t f);
		
		// start the timer
		void start();
		
		// stop the timer
		void stop();
		
		// attach an interrupt handler to timer passing a parameter
		template<class T>void attachInterrupt(void (*isr)(T &param), T &param)
			{ _attachInterrupt((FISHINOTIMER_INTERRUPT_HANDLER)isr, (void *)&param); }

		// attach an interrupt handler to timer with no parameters
		void attachInterrupt(void (*isr)(void))
			{ _attachInterrupt((FISHINOTIMER_INTERRUPT_HANDLER)isr, NULL); }
		
		// detach the interrupt handler from timer
		void detachInterrupt();
		
		// set interrupt priority
		void setInterruptPriority(int pri);
		
		// enable interrupts
		void enableInterrupts(void);
		
		// disable interrupts
		void disableInterrupts(void);
		
		// set timer's clock source
		void setClockSource(FishinoTimerSource src);
		
		// set timer's prescaler
		bool setPrescaler(int ps);
		bool setPrescaler(FishinoTimerPrescale ps);
		
		// set timer's period
		// timer value depends on clock frequency
		void setPeriod(uint32_t per);
		
		// set timer's period in microseconds
		bool setPeriodMicroseconds(uint32_t us);
		
		// set timer's period in milliseconds
		bool setPeriodMilliseconds(uint32_t ms);
		
		// reset timer
		void reset();
		
		// get current timer ticks
		uint32_t getCount();
		
		// get current timer ticks and reset it
		uint32_t getAndResetCount();
		
		void enableGate();
		void disableGate();
		
		uint32_t getRegisters(uint32_t *addr);
};

extern FishinoTimerClass &_timer1(void);
#define Timer1 _timer1()

extern FishinoTimerClass &_timer2(void);
#define Timer2 _timer2()

extern FishinoTimerClass &_timer3(void);
#define Timer3 _timer3()

extern FishinoTimerClass &_timer4(void);
#define Timer4 _timer4()

extern FishinoTimerClass &_timer5(void);
#define Timer5 _timer5()

extern FishinoTimerClass &_timer23(void);
#define Timer23 _timer23()

extern FishinoTimerClass &_timer45(void);
#define Timer45 _timer45()
