//////////////////////////////////////////////////////////////////////////////////
//																				//
//			flash.c -- Flash Memory Operation Functions							//
//																				//
//////////////////////////////////////////////////////////////////////////////////
//	Author: Gene Apperson													*/
//	Copyright 2012, Digilent Inc.												*/
//////////////////////////////////////////////////////////////////////////////////
//																				//
//	This library is free software; you can redistribute it and/or				//
//	modify it under the terms of the GNU Lesser General Public					//
//	License as published by the Free Software Foundation; either				//
//	version 2.1 of the License, or (at your option) any later version.			//
//																				//
//	This library is distributed in the hope that it will be useful,				//
//	but WITHOUT ANY WARRANTY; without even the implied warranty of				//
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.//*	See the GNU			//
//	Lesser General Public License for more details.								//
//																				//
//	You should have received a copy of the GNU Lesser General					//
//	Public License along with this library; if not, write to the				//
//	Free Software Foundation, Inc., 59 Temple Place, Suite 330,					//
//	Boston, MA	02111-1307	USA													//
//																				//
//////////////////////////////////////////////////////////////////////////////////
// Revision History:															//
//																				//
// 08/27/2012 <Gene Apperson> Created											//
//																				//
//////////////////////////////////////////////////////////////////////////////////
#include <Arduino.h>

#ifdef _FISHINO_PIC32_

#include <p32xxxx.h>
#include <stdint.h>
#include <sys/kmem.h>

#include "pic32_flash.h"

// those are addresses, NOT variables nor constants
// they must be used with &operator
extern const uint8_t _eeprom_start;
extern const uint8_t _eeprom_size;
extern const uint8_t _flash_page_size;
extern const uint8_t _flash_row_size;

const uint32_t	EEPROM_START	= (uint32_t)&_eeprom_start;
const size_t	EEPROM_SIZE		= (size_t)(&_eeprom_size);
const size_t	FLASH_PAGE_SIZE	= (size_t)(&_flash_page_size);
const size_t	FLASH_ROW_SIZE	= (size_t)(&_flash_row_size);

#if defined(__PIC32MZ__)
#define NVMDATA     NVMDATA0
#define NVMCON_WREN _NVMCON_WREN_MASK
#define NVMCON_WR   _NVMCON_WR_MASK
#endif

//////////////////////////////////////////////////////////////////////////
// erase a flash page
//	Parameters:
//		adr:	address of page to erase
//
//	Return value:
//		0			if operation succeeded
//		non-zero	if operation failed
uint32_t eraseFlashPage(void *adr)
{
	uint32_t		st;

	// Convert the given address into a physical address
	NVMADDR = KVA_TO_PA((unsigned int) adr);

	// Perform the erase operation.
	st = _doNvmOp(nvmopErasePage);

	return st;
}

//////////////////////////////////////////////////////////////////////////
// Write the specified word to the flash memory at the specified address
//	Parameters:
//		adr:	address of word to write
//		val:	word (32 bit) value
//
//	Return value:
//		0			if operation succeeded
//		non-zero	if operation failed
uint32_t writeFlashWord(void *adr, uint32_t val)
{
	uint32_t	st;

	// Convert the given address into a physical address
	NVMADDR = KVA_TO_PA((unsigned int) adr);

	// Place the data in the NVM data register in preparation for writing.
    NVMDATA = val;

	// Perform the write operation.
	st = _doNvmOp(nvmopWriteWord);

	return st;
}

//////////////////////////////////////////////////////////////////////////
// Write a row of flash at once
//	Parameters:
//		adr:	address of row to write
//		src:	address of buffer containing the row data
//
//	Return value:
//		0			if operation succeeded
//		non-zero	if operation failed
uint32_t writeFlashRow(void *adr, void const *src)
{
	uint32_t	st;

	// Convert the given address into a physical address
	NVMADDR = KVA_TO_PA((unsigned int) adr);
	
	NVMSRCADDR = KVA_TO_PA((unsigned int) src);

	// Perform the write operation.
	st = _doNvmOp(nvmopWriteRow);

	return st;
}

//////////////////////////////////////////////////////////////////////////
// Write a page of flash at once
//	Parameters:
//		adr:	address of page to write
//		src:	address of buffer containing the row data
//
//	Return value:
//		0			if operation succeeded
//		non-zero	if operation failed
uint32_t writeFlashPage(void *adr, void const *src)
{
	// calculate the number of rows in a page
	size_t nRows = FLASH_PAGE_SIZE / FLASH_ROW_SIZE;

	uint32_t rAddr = (uint32_t)adr & ~(FLASH_PAGE_SIZE - 1);
	uint32_t rSrc = (uint32_t)src;
	
	size_t i;
	for(i = 0; i < nRows; i++)
	{
		// Convert the given address into a physical address
		NVMADDR = KVA_TO_PA(rAddr);
		
		// Convert the source address into a physical address
		NVMSRCADDR = KVA_TO_PA(rSrc);
	
		// Perform the write operation.
		if(_doNvmOp(nvmopWriteRow))
			return 1;
		
		rAddr += FLASH_ROW_SIZE;
		rSrc += FLASH_ROW_SIZE;
	}

	return 0;
}


//////////////////////////////////////////////////////////////////////////
// Clear the error status in the NVM controller
//	Parameters:
//		nvmop		- NVM operation to perform
//
//	Return value:
//		0			if operation succeeded
//		non-zero	if operation failed
uint32_t clearNvmError()
{
	return _doNvmOp(nvmopNop);
}

//////////////////////////////////////////////////////////////////////////
// Perform an operation on the flash memory.
//	Parameters:
//		none
//
//	Return value:
//		0			if operation succeeded
//		non-zero	if operation failed
uint32_t __attribute__((nomips16)) _doNvmOp(uint32_t nvmop)
{
//	int			nvmSt;
	int			intSt;
	uint32_t	tm;

	// M00TODO: When DMA operations are supported in the core, need
	// to add code here to suspend DMA during the NVM operation.

	intSt = disableInterrupts();

	// Store the operation code into the NVMCON register.
	NVMCON = NVMCON_WREN | nvmop;

	// We need to wait at least 6uS before performing the operation.
	// We can use the core timer to determine elapsed time based on
	// the CPU operating frequency.
    {
        tm = _CP0_GET_COUNT();
        while (_CP0_GET_COUNT() - tm < ((F_CPU * 6) / 2000000));
    }

	// Unlock so that we can perform the operation.
    NVMKEY 		= 0xAA996655;
    NVMKEY 		= 0x556699AA;
    NVMCONSET 	= NVMCON_WR;

    // Wait for WR bit to clear indicating that the operation has completed.
    while (NVMCON & NVMCON_WR)
        ;

    // Clear the write enable bit in NVMCON to lock the flash again.
    NVMCONCLR = NVMCON_WREN;

	// M00TODO: Resume a suspended DMA operation

	restoreInterrupts(intSt);

	// Return the success state of the operation.
    return(isNvmError());
}

#endif