//////////////////////////////////////////////////////////////////////////////////////
//																					//
//								FishinoTimer8.cpp									//
//	Library to handle Fishino's timers in uniform way between 8 and 32 bit boards	//
//					Created by Massimo Del Fedele, 2019								//
//																					//
//  Copyright (c) 2019 Massimo Del Fedele.  All rights reserved.					//
//																					//
//	Redistribution and use in source and binary forms, with or without				//
//	modification, are permitted provided that the following conditions are met:		//
//																					//
//	- Redistributions of source code must retain the above copyright notice,		//
//	  this list of conditions and the following disclaimer.							//
//	- Redistributions in binary form must reproduce the above copyright notice,		//
//	  this list of conditions and the following disclaimer in the documentation		//
//	  and/or other materials provided with the distribution.						//
//																					//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"		//
//	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE		//
//	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE		//
//	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE		//
//	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR				//
//	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF			//
//	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS		//
//	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN			//
//	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)			//
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE		//
//	POSSIBILITY OF SUCH DAMAGE.														//
//																					//
//	VERSION 7.6.0 - INITIAL VERSION													//
//																					//
//////////////////////////////////////////////////////////////////////////////////////
#include <FishinoTimer.h>

#if defined(_FISHINO_UNO_) || defined(_FISHINO_UNO_R2_) || defined(_FISHINO_MEGA_)  || defined(_FISHINO_MEGA_R2_) || defined(_FISHINO_GUPPY_)

static FishinoTimerClass *_timer1_instance = NULL;
ISR(TIMER1_COMPA_vect)
{
	_timer1_instance->isr();
}

#if defined(_FISHINO_MEGA_) || defined(_FISHINO_MEGA_R2_)

	static FishinoTimerClass *_timer3_instance = NULL;
	ISR(TIMER3_COMPA_vect)
	{
		_timer3_instance->isr();
	}
	
	static FishinoTimerClass *_timer4_instance = NULL;
	ISR(TIMER4_COMPA_vect)
	{
		_timer4_instance->isr();
	}
	
	static FishinoTimerClass *_timer5_instance = NULL;
	ISR(TIMER5_COMPA_vect)
	{
		_timer5_instance->isr();
	}

#endif

///////////////////////////////////////////////////////////////////////////////////////

// internal interrupt handler
void FishinoTimerClass::isr(void)
{
	// if we've set an handler, just call it
	if(_interruptHandler)
		_interruptHandler(_handlerParameter);
	
	// clear interrupt flag on exit
	_tifr->val = 0;
}

// constructor
FishinoTimerClass::FishinoTimerClass()
{
	// Constructor code here
	_prescale = 1;
	_source = FISHINOTIMER_SOURCE_INTERNAL;
	
	_timer_control = NULL;
	_timer_data = NULL;
	_timsk = NULL;
	_tifr = NULL;
	_interruptHandler = NULL;
	_interruptsEnabled = false;
	_running = false;
}

// destructor
FishinoTimerClass::~FishinoTimerClass()
{
	// just in case...
	disableInterrupts();
}

// setup prescaler and source from internal values
void FishinoTimerClass::_setPrescalerAndSource(void)
{
	if(!_running)
	{
		_timer_control->tccrb.cs = 0;
		return;
	}
	switch(_source)
	{
		case FISHINOTIMER_SOURCE_INTERNAL:
			if(_prescale == 1)
				_timer_control->tccrb.cs = 1;
			else if(_prescale == 8)
				_timer_control->tccrb.cs = 2;
			else if(_prescale == 64)
				_timer_control->tccrb.cs = 3;
			else if(_prescale == 256)
				_timer_control->tccrb.cs = 4;
			else if(_prescale == 1024)
				_timer_control->tccrb.cs = 5;
			else
				return;
			break;
		case FISHINOTIMER_SOURCE_EXTERNAL_FALLING:
			_timer_control->tccrb.cs = 6;
			break;
		case FISHINOTIMER_SOURCE_EXTERNAL_RISING:
			_timer_control->tccrb.cs = 7;
			break;
	}
}

// set counter value
void FishinoTimerClass::_setCounter(uint16_t val)
{
	noInterrupts();
	_timer_data->tcnth = val >> 8;
	_timer_data->tcntl = val;
	interrupts();
}

// read counter value
uint16_t FishinoTimerClass::_getCounter(void)
{
	noInterrupts();
	uint16_t val = _timer_data->tcntl;
	val = val | ((uint16_t)(_timer_data->tcnth) << 8);
	interrupts();
	return val;
}

// set top value
void FishinoTimerClass::_setTop(uint16_t val)
{
	noInterrupts();
	_timer_data->ocrah = val >> 8;
	_timer_data->ocral = val;
	interrupts();
}

// set timer frequenency
bool FishinoTimerClass::setFrequency(uint32_t f)
{
	uint32_t baseclock = F_CPU;
	uint8_t ps = 1;
	
	// calculate prescaler value
	while(baseclock / f > 65535)
	{
		baseclock >>= 1;
		ps <<= 1;
	}
	
	// out of range ?
	if(ps > 1024)
		return false;
	
	// round to available prescaler values
	// 1, 8, 64, 256, 1024
	if(ps == 2 || ps == 16)
	{
		// use 8 or 64
		baseclock >>= 2;
		ps <<= 2;
	}
	else if(ps == 4 || ps == 32 || ps == 128 || ps == 512)
	{
		// use 8, 64, 256 or 1024
		baseclock >>= 1;
		ps <<= 1;
	}

	// setup prescaler and source
	_setPrescalerAndSource();
	
	// setup top value
	_setTop(baseclock / f);
	
	return true;
}

// start the timer
void FishinoTimerClass::start()
{
	if(_running)
		return;
	_running = true;
	_setPrescalerAndSource();
}

// stop the timer
void FishinoTimerClass::stop()
{
	_running = false;
	_timer_control->tccrb.cs = 0;
}

// attach an interrupt handler to timer
void FishinoTimerClass::_attachInterrupt(FISHINOTIMER_INTERRUPT_HANDLER isr, void *param)
{
	// if interrupts enabled, disable them
	if(_interruptsEnabled)
		disableInterrupts();
	
	// store user's interrupt handler
	_handlerParameter = param;
	_interruptHandler = isr;
	
	// enable interrupts again
	enableInterrupts();
}

// detach the interrupt handler from timer
void FishinoTimerClass::detachInterrupt()
{
	// if interrupts enabled, disable them
	if(_interruptsEnabled)
		disableInterrupts();
	
	// remove user's interrupt handler
	_interruptHandler = NULL;
	
}

// set interrupt priority
void FishinoTimerClass::setInterruptPriority(int pri)
{
	// nope
	return;
}
		
// enable interrupts
void FishinoTimerClass::enableInterrupts(void)
{
	// if we don't have any handler or already enabled, just do nothing
	if(!_interruptHandler || _interruptsEnabled)
		return;
	
	// clear interrupt flag before enabling
	_tifr->val = 0;

	// enable output compare A match
	_timsk->ociea = 1;
	
	// mark interrupts as enabled
	_interruptsEnabled = true;
}

// disable interrupts
void FishinoTimerClass::disableInterrupts(void)
{
	// if not enabled, just do nothing
	if(!_interruptsEnabled)
		return;
	
	// disable interrupts
	_timsk->ociea = 0;
	
	// clear interrupt flag
	_tifr->val = 0;

	// mark interrupts as disabled
	_interruptsEnabled = false;
}

// set timer's clock source
void FishinoTimerClass::setClockSource(FishinoTimerSource src)
{
	_source = src;
	_setPrescalerAndSource();
}

// set timer's prescaler
bool FishinoTimerClass::setPrescaler(FishinoTimerPrescale ps)
{
	switch(ps)
	{
		case FISHINOTIMER_PRESCALE_1:
			_prescale = 1;
			break;
				
		case FISHINOTIMER_PRESCALE_8:
			_prescale = 8;
			break;
				
		case FISHINOTIMER_PRESCALE_64:
			_prescale = 64;
			break;
				
		case FISHINOTIMER_PRESCALE_256:
			_prescale = 256;
			break;
				
		case FISHINOTIMER_PRESCALE_1024:
			_prescale = 1024;
			break;
				
			// we shall maybe signal an error here...
		default:
			return false;
	}
	_setPrescalerAndSource();
	return true;
}

bool FishinoTimerClass::setPrescaler(int ps)
{
	switch(ps)
	{
		case 1:
		case 8:
		case 64:
		case 256:
		case 1024:
			_prescale = ps;
			break;
			
		default:
			return false;
	}
	_setPrescalerAndSource();
	return true;
}


// set timer's period
// timer value depends on clock frequency
void FishinoTimerClass::setPeriod(uint32_t per)
{
	_setTop(per);
}

// set timer's period in microseconds
bool FishinoTimerClass::setPeriodMicroseconds(uint32_t us)
{
	// resolution is 1 uSecond, not less
	if(!us)
		us = 1;
	
	uint64_t per = (uint64_t)F_CPU * us;
	per /= 1000000;
	uint32_t div = 1;
	while(per > (uint16_t)-1)
	{
		per >>= 1;
		div <<= 1;
	}

	// check if can't cope with this period
	if(div > 1024)
		return false;
	
	// round to available prescaler values
	// 1, 8, 64, 256, 1024
	if(div == 2 || div == 16)
	{
		// use 8 or 64
		per >>= 2;
		div <<= 2;
	}
	else if(div == 4 || div == 32 || div == 128 || div == 512)
	{
		// use 8, 64, 256 or 1024
		per >>= 1;
		div <<= 1;
	}
	_prescale = div;
	
	// set top value
	_setTop(per);

	// setup prescaler and source
	_setPrescalerAndSource();

	// all ok
	return true;
}

// set timer's period in milliseconds
bool FishinoTimerClass::setPeriodMilliseconds(uint32_t ms)
{
	return setPeriodMicroseconds(ms * 1000);
}

// reset timer
void FishinoTimerClass::reset()
{
	_setCounter(0);
}

// get current timer ticks
uint32_t FishinoTimerClass::getCount()
{
	return _getCounter();
}

// get current timer ticks and reset it
uint32_t FishinoTimerClass::getAndResetCount()
{
	uint32_t t = _getCounter();
	_setCounter(0);
	return t;
}

void FishinoTimerClass::enableGate()
{
	// nope
}

void FishinoTimerClass::disableGate()
{
	// nope
}

// get a Register Value
uint32_t FishinoTimerClass::getRegisters(uint32_t *addr)
{
	// nope
	return 0;
}


///////////////////////////////////////////////////////////////////////////////////////
struct _FishinoTimer_Initializer
{
	volatile void *_timer_control;	// timer control bytes
	volatile void *_timer_data;		// timer data bytes
	volatile void *_timsk;			// timer interrupt mask
	volatile void *_tifr;			// timer interrupt flags

	FishinoTimerClass **_instance;
};

const _FishinoTimer_Initializer _FishinoTimer_Initializers[] =
{
	{ &TCCR1A, &TCNT1L, &TIMSK1, &TIFR1, &_timer1_instance },
#if defined(_FISHINO_MEGA_) || defined(_FISHINO_MEGA_R2_)
	{ &TCCR3A, &TCNT3L, &TIMSK3, &TIFR3, &_timer3_instance },
	{ &TCCR4A, &TCNT4L, &TIMSK4, &TIFR4, &_timer4_instance },
	{ &TCCR5A, &TCNT5L, &TIMSK5, &TIFR5, &_timer5_instance }
#endif
};

void _FishinoTimer_Initialize(FishinoTimerClass &t, uint8_t index)
{
	_FishinoTimer_Initializer const &i = _FishinoTimer_Initializers[index];
	
	t._timer_control	= (volatile _timer_control_s *)i._timer_control;
	t._timer_data		= (volatile _timer_data_s *)i._timer_data;
	t._timsk			= (volatile _timsk_s *)i._timsk;
	t._tifr				= (volatile _tifr_s *)i._tifr;
	
	*i._instance = &t;
	
	// clear interrupts and disable them
	t._timsk->val = 0;
	t._tifr->val = 0;

	// setup OC mode
	t._timer_control->tccra.wgm = 0;
	t._timer_control->tccrb.wgm = 1;
	
	// set some meaningful values on top
	t._setTop(128);
}

#define FISHINOTIMER_DEF(name, idx) \
FishinoTimerClass &_timer##name() \
{ \
	static FishinoTimerClass t; \
	static bool initialized = false; \
	if(!initialized) \
	{ \
		_FishinoTimer_Initialize(t, idx); \
		initialized = true; \
	} \
	return t; \
}

FISHINOTIMER_DEF(1, 0)
FISHINOTIMER_DEF(2, 1)

#if defined(_FISHINO_MEGA_) || defined(_FISHINO_MEGA_R2_)
FISHINOTIMER_DEF(3, 2)
FISHINOTIMER_DEF(4, 3)
FISHINOTIMER_DEF(5, 4)
#endif

#undef FISHINOTIMER_DEF

#endif
