//////////////////////////////////////////////////////////////////////////////////////
//																					//
//								FishinoTimer8.h										//
//	Library to handle Fishino's timers in uniform way between 8 and 32 bit boards	//
//					Created by Massimo Del Fedele, 2019								//
//																					//
//  Copyright (c) 2019 Massimo Del Fedele.  All rights reserved.					//
//																					//
//	Redistribution and use in source and binary forms, with or without				//
//	modification, are permitted provided that the following conditions are met:		//
//																					//
//	- Redistributions of source code must retain the above copyright notice,		//
//	  this list of conditions and the following disclaimer.							//
//	- Redistributions in binary form must reproduce the above copyright notice,		//
//	  this list of conditions and the following disclaimer in the documentation		//
//	  and/or other materials provided with the distribution.						//
//																					//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"		//
//	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE		//
//	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE		//
//	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE		//
//	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR				//
//	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF			//
//	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS		//
//	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN			//
//	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)			//
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE		//
//	POSSIBILITY OF SUCH DAMAGE.														//
//																					//
//	VERSION 7.6.0 - INITIAL VERSION													//
//																					//
//////////////////////////////////////////////////////////////////////////////////////
typedef void (*FISHINOTIMER_INTERRUPT_HANDLER)(void *param);

extern "C" void TIMER1_COMPA_vect(void);
extern "C" void TIMER2_COMPA_vect(void);
#if defined(_FISHINO_MEGA_) || defined(_FISHINO_MEGA_R2_)
	extern "C" void TIMER3_COMPA_vect(void);
	extern "C" void TIMER4_COMPA_vect(void);
	extern "C" void TIMER5_COMPA_vect(void);
#endif

enum FishinoTimerSource
{
	FISHINOTIMER_SOURCE_INTERNAL			= 0,
	FISHINOTIMER_SOURCE_EXTERNAL			= 1,
	FISHINOTIMER_SOURCE_EXTERNAL_RISING		= 1,
	FISHINOTIMER_SOURCE_EXTERNAL_FALLING	= 2
};

enum FishinoTimerPrescale
{
	FISHINOTIMER_PRESCALE_1,
	FISHINOTIMER_PRESCALE_8,
	FISHINOTIMER_PRESCALE_64,
	FISHINOTIMER_PRESCALE_256,
	FISHINOTIMER_PRESCALE_1024
};

struct _timer_data_s
{
	uint8_t tcntl;
	uint8_t tcnth;
	uint8_t icrl;
	uint8_t icrh;
	uint8_t ocral;
	uint8_t ocrah;
	uint8_t ocrbl;
	uint8_t ocrbh;
	uint8_t ocrcl;
	uint8_t ocrch;
} __attribute__((packed));

struct _tccra_s
{
	union
	{
		uint8_t val;
		struct
		{
			unsigned wgm:2;
			unsigned comc:2;
			unsigned comb:2;
			unsigned coma:2;
		};
	};
} __attribute__((packed));

struct _tccrb_s
{
	union
	{
		uint8_t val;
		struct
		{
			unsigned cs:3;
			unsigned wgm:2;
			unsigned _unused:1;
			unsigned ices:1;
			unsigned icnc:1;
		};
	};
} __attribute__((packed));

struct _tccrc_s
{
	union
	{
		uint8_t val;
		struct
		{
			unsigned _unused:5;
			unsigned focc:1;
			unsigned focb:1;
			unsigned foca:1;
		};
	};
} __attribute__((packed));

struct _timer_control_s
{
	struct _tccra_s tccra;
	struct _tccrb_s tccrb;
	struct _tccrc_s tccrc;
} __attribute__((packed));

struct _timsk_s
{
	union
	{
		uint8_t val;
		struct
		{
			unsigned toie:1;
			unsigned ociea : 1;
			unsigned ocieb : 1;
			unsigned ociec : 1;
			unsigned _unused1:1;
			unsigned icie:1;
			unsigned _unused2:2;
		};
	};
} __attribute__((packed));

struct _tifr_s
{
	union
	{
		uint8_t val;
		struct
		{
			unsigned tov:1;
			unsigned ocfa : 1;
			unsigned ocfb : 1;
			unsigned ocfc : 1;
			unsigned _unused1:1;
			unsigned icf:1;
			unsigned _unused2:2;
		};
	};
} __attribute__((packed));

class FishinoTimerClass
{
	friend void _FishinoTimer_Initialize(FishinoTimerClass &t, uint8_t index);

	private:

		friend void TIMER1_COMPA_vect(void);

	#if defined(_FISHINO_MEGA_) || defined(_FISHINO_MEGA_R2_)
		friend void TIMER3_COMPA_vect(void);
		friend void TIMER4_COMPA_vect(void);
		friend void TIMER5_COMPA_vect(void);
	#endif

		volatile struct _timer_control_s	*_timer_control;		// timer control bytes
		volatile struct _timer_data_s		*_timer_data;			// timer data bytes
		volatile struct _timsk_s			*_timsk;	// timer interrupt mask
		volatile struct _tifr_s				*_tifr;		// timer interrupt flags
		
		uint16_t					_prescale;	// prescaler value (1, 8, 64, 256, 1024)
		uint8_t						_source;	// clock source (INTERNAL, EXTERNAL_RISING, EXTERNAL_FALLING)
		
		bool						_running;
		
		// setup prescaler and source from internal values
		void _setPrescalerAndSource(void);
		
		// set counter value
		void _setCounter(uint16_t val);
		
		// read counter value
		uint16_t _getCounter(void);
		
		// set top value
		void _setTop(uint16_t val);

		// external interrupt handler pointer
		// used to set PIC interrupt vector
		void (*_isrHandler)(void);
		
		// internal interrupt handler
		void isr(void);
		
		// user interrupt handler -- he can pass a parameter
		// stored previously by attachInterrupt() call
		void *_handlerParameter;
		FISHINOTIMER_INTERRUPT_HANDLER _interruptHandler;
		
		// flag for enabled interrupts
		bool _interruptsEnabled;
		
		// helper for parameter-safe attachinterrupt
		void _attachInterrupt(FISHINOTIMER_INTERRUPT_HANDLER handler, void *param);

	public:
		// constructor
		FishinoTimerClass();
		
		// destructor
		~FishinoTimerClass();
		
		// set timer frequenency
		bool setFrequency(uint32_t f);
		
		// start the timer
		void start();
		
		// stop the timer
		void stop();
		
		// attach an interrupt handler to timer passing a parameter
		template<class T>void attachInterrupt(void (*isr)(T &param), T &param)
			{ _attachInterrupt((FISHINOTIMER_INTERRUPT_HANDLER)isr, (void *)&param); }

		// attach an interrupt handler to timer with no parameters
		void attachInterrupt(void (*isr)(void))
			{ _attachInterrupt((FISHINOTIMER_INTERRUPT_HANDLER)isr, NULL); }
		
		// detach the interrupt handler from timer
		void detachInterrupt();
		
		// set interrupt priority
		void setInterruptPriority(int pri);
		
		// enable interrupts
		void enableInterrupts(void);
		
		// disable interrupts
		void disableInterrupts(void);
		
		// set timer's clock source
		void setClockSource(FishinoTimerSource src);
		
		// set timer's prescaler
		bool setPrescaler(int ps);
		bool setPrescaler(FishinoTimerPrescale ps);
		
		// set timer's period
		// timer value depends on clock frequency
		void setPeriod(uint32_t per);
		
		// set timer's period in microseconds
		bool setPeriodMicroseconds(uint32_t us);
		
		// set timer's period in milliseconds
		bool setPeriodMilliseconds(uint32_t ms);
		
		// reset timer
		void reset();
		
		// get current timer ticks
		uint32_t getCount();
		
		// get current timer ticks and reset it
		uint32_t getAndResetCount();
		
		void enableGate();
		void disableGate();
		
		uint32_t getRegisters(uint32_t *addr);
};

extern FishinoTimerClass &_timer1(void);
#define Timer1 _timer1()

#if defined(_FISHINO_MEGA_) || defined(_FISHINO_MEGA_R2_)
	extern FishinoTimerClass &_timer3(void);
	#define Timer3 _timer3()

	extern FishinoTimerClass &_timer4(void);
	#define Timer4 _timer4()

	extern FishinoTimerClass &_timer5(void);
	#define Timer5 _timer5()
#endif
