//////////////////////////////////////////////////////////////////////////////////////
//						 		MotorFish Firmware									//
//																					//
//			Library to handle MotorFish closed loop stepper driver board			//
//																					//
//		Copyright (c) 2019 Massimo Del Fedele.  All rights reserved.				//
//																					//
//	Redistribution and use in source and binary forms, with or without				//
//	modification, are permitted provided that the following conditions are met:		//
//																					//
//	- Redistributions of source code must retain the above copyright notice,		//
//	  this list of conditions and the following disclaimer.							//
//	- Redistributions in binary form must reproduce the above copyright notice,		//
//	  this list of conditions and the following disclaimer in the documentation		//
//	  and/or other materials provided with the distribution.						//
//																					//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"		//
//	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE		//
//	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE		//
//	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE		//
//	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR				//
//	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF			//
//	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS		//
//	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN			//
//	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)			//
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE		//
//	POSSIBILITY OF SUCH DAMAGE.														//
//																					//
//  Version 7.8.0 -- March 2019		Initial version									//
//																					//
//////////////////////////////////////////////////////////////////////////////////////
#include <FishinoFlash.h>
#include <FishinoEEPROM.h>

#include<MotorFish.h>

#define DEBUG_LEVEL_INFO
#include <FishinoDebug.h>

// select which serial port to use
// Serial is USB port
// Serial0 is Serial connected to SDI/SDO
#define MOTORFISH_SERIAL Serial

// eeprom map for calibration and configuration data
#define MOTORFISH_EEPROM_START				0x0

#define MOTORFISH_CALIBRATION_START			MOTORFISH_EEPROM_START
#define MOTORFISH_CALIBRATION_LEN			4096

#define MOTORFISH_SETUP_EEPROM				(MOTORFISH_CALIBRATION_START + MOTORFISH_CALIBRATION_LEN)

#define MOTORFISH_EEPROM_MAGIC				MOTORFISH_SETUP_EEPROM
#define MOTORFISH_EEPROM_MAGIC_LEN			4
#define MOTORFISH_EEPROM_MAGIC_VAL			0xAA55A55A

#define MOTORFISH_MODE						(MOTORFISH_EEPROM_MAGIC + MOTORFISH_EEPROM_MAGIC_LEN)
#define MOTORFISH_MODE_LEN					1

#define MOTORFISH_MOTOR_DIRECTION			(MOTORFISH_MODE + MOTORFISH_MODE_LEN)
#define MOTORFISH_MOTOR_DIRECTION_LEN		1

#define MOTORFISH_ENCODER_PLACEMENT			(MOTORFISH_MOTOR_DIRECTION + MOTORFISH_MOTOR_DIRECTION_LEN)
#define MOTORFISH_ENCODER_PLACEMENT_LEN		1

#define MOTORFISH_DUMMY1					(MOTORFISH_ENCODER_PLACEMENT + MOTORFISH_ENCODER_PLACEMENT_LEN)
#define MOTORFISH_DUMMY1_LEN				1

#define MOTORFISH_KP						(MOTORFISH_DUMMY1 + MOTORFISH_DUMMY1_LEN)
#define MOTORFISH_KP_LEN					2

#define MOTORFISH_KI						(MOTORFISH_KP + MOTORFISH_KP_LEN)
#define MOTORFISH_KI_LEN					2

#define MOTORFISH_KD						(MOTORFISH_KI + MOTORFISH_KI_LEN)
#define MOTORFISH_KD_LEN					2

#define MOTORFISH_INTEGRALDAMP				(MOTORFISH_KD + MOTORFISH_KD_LEN)
#define MOTORFISH_INTEGRALDAMP_LEN			2

#define MOTORFISH_SYNCTIME					(MOTORFISH_INTEGRALDAMP + MOTORFISH_INTEGRALDAMP_LEN)
#define MOTORFISH_SYNCTIME_LEN				4

#define MOTORFISH_WINDING_CURRENT			(MOTORFISH_SYNCTIME + MOTORFISH_SYNCTIME_LEN)
#define MOTORFISH_WINDING_CURRENT_LEN		sizeof(double)

// used I/O pins
#define STEP			6
#define DIR				7
#define ENABLE			8

// leds
#define YELLOW			PIN_LED1
#define BLUE			PIN_LED2
#define RED				PIN_LED3
#define WHITE			PIN_LED4

// quick access to above ports
#define DIR_PORT		PORTD
#define DIR_BIT			8

#define ENABLE_PORT		PORTD
#define ENABLE_BIT		0

#define YELLOW_PORT		PORTD
#define YELLOW_LAT		LATD
#define YELLOW_BIT		3

#define BLUE_PORT		PORTD
#define BLUE_LAT		LATD
#define BLUE_BIT		1

#define RED_PORT		PORTC
#define RED_LAT			LATC
#define RED_BIT			13

#define WHITE_PORT		PORTC
#define WHITE_LAT		LATC
#define WHITE_BIT		14


FlashStringArray OperatingModes = FStringArray (
	"DISABLED",
	"STEPPER",
	"CLOSED LOOP"
);

FlashStringArray EncoderPlacements = FStringArray (
	"REAR",
	"FRONT"
);

FlashStringArray MotorDirections = FStringArray (
	"CLOCKWISE",
	"COUNTERCLOCKWISE"
);

void printMode(Stream &s, FlashStringArray const &names, int current)
{
	int n = names.getCount();
	for(int i = 0; i < n; i++)
	{
		if(current == i)
			s << "*";
		s << names[i];
		if(current == i)
			s << "*";
		if(i < n - 1)
			s << ",";
	}
}

void waitKey(void)
{
	MOTORFISH_SERIAL << "Press a key to continue...";
	while(!MOTORFISH_SERIAL.available())
		;
	MOTORFISH_SERIAL.read();
	MOTORFISH_SERIAL << "\n";
	return;
}

// read an int16_t from serial port (-32767..32767)
bool readInt16(const char *prompt, int16_t &val)
{
	// previous value to put in prompt, accepted with ENTER
	int16_t prevVal = val;
	char buf[20];
	int iBuf = 0;
	buf[0] = 0;
	
	MOTORFISH_SERIAL << prompt << " (" << prevVal << ") : ";
	while(true)
	{
		// wait for a character from serial port
		while(!MOTORFISH_SERIAL.available())
			;
		
		// we accept only '0'..'9', '-', BS, space, ESC and return
		char c = MOTORFISH_SERIAL.read();
		if(c == '-' && iBuf == 0)
		{
			buf[iBuf++] = c;
			buf[iBuf] = 0;
			MOTORFISH_SERIAL.write(c);
		}
		else if(c >= '0' && c <= '9' && iBuf < 19)
		{
			buf[iBuf++] = c;
			buf[iBuf] = 0;
			MOTORFISH_SERIAL.write(c);
		}
		else if(c == 0x08 && iBuf > 0)
		{
			MOTORFISH_SERIAL.println();
			MOTORFISH_SERIAL << prompt << " (" << prevVal << ") : ";
			buf[--iBuf] = 0;
			for(int i = 0; i < iBuf; i++)
				MOTORFISH_SERIAL.write(buf[i]);
		}
		else if(c == ' ' || c == '\n')
		{
			if(!iBuf)
			{
				val = prevVal;
				MOTORFISH_SERIAL.println(prevVal);
				return true;
			}
			
			int nVal;
			if(sscanf(buf, "%d", &nVal) == 1 && nVal >= -32767 && nVal <= 32767)
			{
				val = (int16_t)nVal;
				return true;
			}
			MOTORFISH_SERIAL << "\n*" << buf << "* invalid\n";
			MOTORFISH_SERIAL << prompt << " (" << prevVal << ") : ";
			iBuf = 0;
			buf[0] = 0;
		}
		else if(c == 27)
		{
			val = prevVal;
			MOTORFISH_SERIAL << "*CANCEL*\n";
			return false;
		}
	} // while(true)
}

// read an double from serial port
bool readDouble(const char *prompt, double &val)
{
	// previous value to put in prompt, accepted with ENTER
	double prevVal = val;
	char buf[20];
	int iBuf = 0;
	buf[0] = 0;
	
	MOTORFISH_SERIAL << prompt << " (" << prevVal << ") : ";
	while(true)
	{
		// wait for a character from serial port
		while(!MOTORFISH_SERIAL.available())
			;
		
		// we accept only '0'..'9', '-', BS, space, ESC and return
		char c = MOTORFISH_SERIAL.read();
		if(c == '-' && iBuf == 0)
		{
			buf[iBuf++] = c;
			buf[iBuf] = 0;
			MOTORFISH_SERIAL.write(c);
		}
		else if(c == '.' && !strchr(buf, '.'))
		{
			buf[iBuf++] = c;
			buf[iBuf] = 0;
			MOTORFISH_SERIAL.write(c);
		}
		else if(c >= '0' && c <= '9' && iBuf < 19)
		{
			buf[iBuf++] = c;
			buf[iBuf] = 0;
			MOTORFISH_SERIAL.write(c);
		}
		else if(c == 0x08 && iBuf > 0)
		{
			MOTORFISH_SERIAL.println();
			MOTORFISH_SERIAL << prompt << " (" << prevVal << ") : ";
			buf[--iBuf] = 0;
			for(int i = 0; i < iBuf; i++)
				MOTORFISH_SERIAL.write(buf[i]);
		}
		else if(c == ' ' || c == '\n')
		{
			if(!iBuf)
			{
				val = prevVal;
				MOTORFISH_SERIAL.println(prevVal);
				return true;
			}
			
			double nVal;
			if(sscanf(buf, "%lf", &nVal) == 1)
			{
				val = nVal;
				return true;
			}
			MOTORFISH_SERIAL << "\n*" << buf << "* invalid\n";
			MOTORFISH_SERIAL << prompt << " (" << prevVal << ") : ";
			iBuf = 0;
			buf[0] = 0;
		}
		else if(c == 27)
		{
			val = prevVal;
			MOTORFISH_SERIAL << "*CANCEL*\n";
			return false;
		}
	} // while(true)
}

// current working mode
// 0 : disabled
// 1 : stepper
// 2 : closedloop
uint8_t currentMode;

// step interrupt routine in servo mode
void servoStepISR(void)
{
	static bool ledState = false;
	static bool inside = false;
	if(inside)
		return;
	inside = true;
	
	if(ClosedLoop.isEnabled())
	{
		// get direction from direction input
		bool dir = DIR_PORT & _BV(DIR_BIT);
		
		// just blink led by now
		if(ledState)
			WHITE_LAT |= _BV(WHITE_BIT);
		else
			WHITE_LAT &= ~_BV(WHITE_BIT);
		ledState = !ledState;
	
		// execute the step
		ClosedLoop.step(dir);
	}
	
	// clear interrupt flag
	IFS0bits.INT1IF = 0;
	
	inside = false;
}


// step interrupt routine in stepper mode
void stepperStepISR(void)
{
	// avoid reentering
	static bool inside = false;
	if(inside)
		return;
	inside = true;
	
	static bool ledState = false;
	
	bool dir = DIR_PORT & _BV(DIR_BIT);
	
	// just blink led by now
	if(ledState)
		WHITE_LAT |= _BV(WHITE_BIT);
	else
		WHITE_LAT &= ~_BV(WHITE_BIT);
	ledState = !ledState;

	Stepper.step(dir);
	
	IFS0bits.INT1IF = 0;
	
	inside = false;
}

// deactivate all modes
void deactivateAll(void)
{
	// detach step interrupt
	detachInterrupt(digitalPinToInterrupt(STEP));
	
	// disable servo mode
	ClosedLoop.disable();
}

// activate servo mode
void setupServo(void)
{
	// deactivate any previous mode
	deactivateAll();

	// attach step interrupt
	// we use fast interrupts here, not arduino routines
	// so we can react to more steps
	// connect step input to ISR
	attachInterrupt(digitalPinToInterrupt(STEP), servoStepISR, RISING);
	
	// enable servo mode
	ClosedLoop.enable();

	// rise interrupt priority to max, we shall not loose steps
    IPC1bits.INT1IP		=	6;
}

// activate stepper mode
void setupStepper(void)
{
	// deactivate any previous mode
	deactivateAll();
	
	Stepper.setWindingCurrentFactor(65535);
	Stepper.energize(true);
	
	// attach step interrupt
	// we use fast interrupts here, not arduino routines
	// so we can react to more steps
	// connect step input to ISR
	attachInterrupt(digitalPinToInterrupt(STEP), stepperStepISR, RISING);
	
	// rise interrupt priority to max, we shall not loose steps
    IPC1bits.INT1IP		=	6;
}

// setup required mode -- return previous active mode
uint8_t setupMode(uint8_t newMode)
{
	uint8_t prevMode = currentMode;
	currentMode = newMode;
	if(currentMode == 1)
		setupStepper();
	else if(currentMode == 2)
		setupServo();
	else
		deactivateAll();
	return prevMode;
}


// calibration routine
void calibrate(void)
{
	DEBUG_INFO_N("MotorFish Calibration\n\n");
	
	// disable calibration, we're building it!
	Encoder.disableCalibration();

	// turn on the stepper motor
	Stepper.energize(true);
	Stepper.setWindingCurrentFactor(65535);
	delay(100);
	
	DEBUG_INFO_N("Locating motor Zero point...");
	bool found = false;
	uint32_t prevPos = Encoder.readSSIPosition();
	for(int i = 0; i < 64*200; i++)
	{
		Stepper.stepForward();
		delay(1);
		uint32_t curPos = Encoder.readSSIPosition();
//		DEBUG_INFO("CurPos:%06d\n", curPos);
		if(curPos < prevPos && prevPos - curPos > 500)
		{
			found = true;
			break;
		}
		prevPos = curPos;
	}
	if(!found)
	{
		DEBUG_ERROR_N("ERROR\nZero point not found!");
		waitKey();
		Encoder.enableCalibration();
		return;
	}
	DEBUG_INFO_N("DONE\n");
	delay(100);


	DEBUG_INFO_N("Measure loop...");
	
	// as we must find a REVERSE relationship between read data and motor position
	// we can't simply fill an array with difference of given and expected position
	// we use 2 arrays to store the read true positions indexed by encoder reading
	// and the number of hits; then we walk back into the array and calculate
	// correction factors. It's memory expensive procedure, so we shall try to optimize
	uint8_t hits[4096];
	float vals[4096];
	for(int i = 0; i < 4096; i++)
	{
		hits[i] = 0;
		vals[i] = 0;
	}
	
	// loop for a full motor turn
	for(int32_t i = 0; i < 64*200; i++)
	{
		// get expected encoder value - 0..4095
		uint16_t truePos = STEP2ENC(i);

		// read true encoder value - 0..4095
		uint16_t encoderPos = Encoder.readSSIPosition();
		
		int delta = (int)truePos - (int)encoderPos;
		if(delta < -2048)
			delta += 4096;
		else if(delta > 2048)
			delta -= 4096;
//		DEBUG_INFO("POS:%06u -- ENC:%06u -- DELTA:%+06d\n", truePos, encoderPos, delta);

		// sum true position into vals array and increment hits count
		// as the values are circular, we must ensure to use the correct
		// value to sum
		if(!hits[encoderPos])
			vals[encoderPos] = truePos;
		else
		{
			double prev = vals[encoderPos] / hits[encoderPos];
			double dist = fabs(prev - (double)truePos);
			if(dist > 2048)
				vals[encoderPos] += (double)truePos - 4096;
			else
				vals[encoderPos] += truePos;
		}
		if(hits[encoderPos] >= 255)
		{
			DEBUG_ERROR("Too many hits at %d", encoderPos);
			waitKey();
			Encoder.enableCalibration();
			return;
		}
		hits[encoderPos]++;

		Stepper.stepForward();
		delay(1);
	}
	
	// we're done with stepper motor
	Stepper.energize(false);


	DEBUG_INFO_N("DONE\n");
	DEBUG_INFO_N("Calculate corrections factors...");

	// now calculate mean positions, leaving out missing hits
	for(int i = 0; i < 4096; i++)
	{
		if(hits[i])
			vals[i] /= hits[i];
	}
	
	// now the difficult part... we must fill the holes
	// as some encoder value may be missing in this run
	// so we interpolate with near hits
	int i = 0;
	while(i < 4096)
	{
		// if already filled, skip
		if(hits[i])
		{
			i++;
			continue;
		}

		// found an hole, look around it
		// find left and right filled items and the count of
		// holes between them
		int left = i;
		while(!hits[left])
		{
			left--;
			if(left < 0)
				left += 4096;
		}
		int right = i;
		while(!hits[right])
		{
			right++;
			if(right >= 4096)
				right -= 4096;
		}
		int nHoles = right - left - 1;
		if(nHoles < 0)
			nHoles += 4096;
		else if(nHoles >= 4096)
			nHoles -= 4096;
		
		// fill the holes
		double d = (vals[right] - vals[left]);
		if(d < -2048)
			d += 4096;
		else if(d > 2048)
			d -= 4096;
		d /= (nHoles + 1);
		double v = vals[left] + d;
		if(v >= 4096)
			v -= 4096;
		int j = (left + 1) % 4096;
		while(j != right)
		{
			vals[j] = v;
			v += d;
			if(v >= 4096)
				v -= 4096;
			j = (j + 1) % 4096;
		}
		
		// skip filled holes
		i = right + 1;
	}
	
	// ok, now we've got a table indexing read positions with true ones
	// we shall compute the differences, round to nearest int and store to eeprom
	// if we find a difference greater than +/- 127 we've got a problem
	for(int i = 0; i < 4096; i++)
	{
		vals[i] -= i;
		if(vals[i] < -2048)
			vals[i] += 4096;
		else if(vals[i] > 2048)
			vals[i] -= 4096;
		if(vals[i] < -127 || vals[i] > 127)
		{
			DEBUG_ERROR("Error too big at position %d, value is %f", i, vals[i]);
			waitKey();
			Encoder.enableCalibration();
			return;
		}
		if(vals[i] < 0)
			vals[i] -= 0.5;
		else
			vals[i] += 0.5;
	}

	DEBUG_INFO_N("DONE\n");
	DEBUG_INFO_N("Writing data to EEPROM...");

	FishinoEEPROM.eraseArea(MOTORFISH_CALIBRATION_START, MOTORFISH_CALIBRATION_LEN);
	int minDiff = 999999;
	int maxDiff = -9999999;
	for(int i = 0; i < 4096; i++)
	{
		// write to eeprom
		int8_t d = (int8_t)vals[i];
		FishinoEEPROM.write8s(MOTORFISH_CALIBRATION_START + i, d);
		if(d < minDiff)
			minDiff = d;
		if(d > maxDiff)
			maxDiff = d;
	}

	DEBUG_INFO_N("\nMin diff : %d, Max diff : %d\n", minDiff, maxDiff);
	DEBUG_INFO_N("DONE\n");
	waitKey();
	Encoder.enableCalibration();
}

// dump calibration values
void dumpCalibration(void)
{
	DEBUG_INFO_N("Calibration values:\n");
	int minDiff = 999999;
	int maxDiff = -9999999;
	for(int i = 0; i < 4096; i++)
	{
		if(!(i % 16))
			DEBUG_INFO_N(" \nPOS:%05d ", i);
		int d = FishinoEEPROM.read8s(i);
		if(d < minDiff)
			minDiff = d;
		if(d > maxDiff)
			maxDiff = d;
		DEBUG_INFO_N("%+04d,", d);
	}
	DEBUG_INFO_N("\nMin diff : %d, Max diff : %d\n", minDiff, maxDiff);
	DEBUG_INFO_N("DONE\n");
	waitKey();
}

void eraseCalibration(void)
{
	DEBUG_INFO_N("Erasing calibration data...");
	FishinoEEPROM.eraseArea(MOTORFISH_CALIBRATION_START, MOTORFISH_CALIBRATION_LEN);
	DEBUG_INFO_N("DONE\n");
	waitKey();
}

void encoderTest(void)
{
	MOTORFISH_SERIAL << "MotorFish Encoder Test\n\n";
	MOTORFISH_SERIAL << "Rotate motor's shaft and look at position\n";
	MOTORFISH_SERIAL << "Press any key to end the test\n\n";
	int32_t curPos = 0;
	int32_t prevPos = -1;
	Encoder.resetMotorPos();
	while(!MOTORFISH_SERIAL.available())
	{
		curPos = Encoder.motorPos();
		if(curPos != prevPos)
		{
			prevPos = curPos;
			DEBUG_INFO("Pos:%8d\n", curPos);
		}
	}
	MOTORFISH_SERIAL.read();
}

void openLoopTest(void)
{
	MOTORFISH_SERIAL << "MotorFish Openloop Test\n\n";
	MOTORFISH_SERIAL << "The motor will rotate as a normal stepper motor\n";
	MOTORFISH_SERIAL << "one turn forward and one turn backward until a key is pressed\n";
	MOTORFISH_SERIAL << "The shaft position will be printed on serial monitor\n";
	MOTORFISH_SERIAL << "Press any key to start...";
	while(!MOTORFISH_SERIAL.available())
		;
	MOTORFISH_SERIAL.read();
	Stepper.energize(true);
	Stepper.setWindingCurrentFactor(65535);
	delay(100);
	Encoder.resetMotorPos();
	int32_t pos = 0;
	while(!MOTORFISH_SERIAL.available())
	{
		for(int i = 0; i < 200 * 64; i++)
		{
			if(MOTORFISH_SERIAL.available())
				break;
			Stepper.stepForward();
			pos++;
			delay(1);
			int32_t mPos = Encoder.motorPos();
			DEBUG_INFO("Pos:%8d - Error:%+03d\n", mPos, mPos - pos);
			
		}
		for(int i = 0; i < 200 * 64; i++)
		{
			if(MOTORFISH_SERIAL.available())
				break;
			Stepper.stepBackward();
			pos--;
			delay(1);
			int32_t mPos = Encoder.motorPos();
			DEBUG_INFO("Pos:%8d - Error:%+03d\n", mPos, mPos - pos);
		}
	}
	Stepper.energize(false);
	MOTORFISH_SERIAL.read();	
}

void closedLoopTest(void)
{
	MOTORFISH_SERIAL << "MotorFish Closedloop Test\n\n";
	MOTORFISH_SERIAL << "The motor will rotate as a servo motor\n";
	MOTORFISH_SERIAL << "one turn forward and one turn backward until a key is pressed\n";
	MOTORFISH_SERIAL << "The shaft position will be printed on serial monitor\n";
	MOTORFISH_SERIAL << "Press any key to start...";
	while(!MOTORFISH_SERIAL.available())
		;
	MOTORFISH_SERIAL.read();

	Encoder.resetMotorPos();

	// enable servo mode
	ClosedLoop.enable();

	int32_t pos = Encoder.motorPos();
	while(!MOTORFISH_SERIAL.available())
	{
		for(int i = 0; i < 200 * 64; i++)
		{
			if(MOTORFISH_SERIAL.available())
				break;
			ClosedLoop.stepForward();
			pos++;
			delay(1);
			int32_t mPos = Encoder.motorPos();
			DEBUG_INFO("Pos:%8d - Error:%+03d\n", mPos, mPos - pos);
			
		}
		for(int i = 0; i < 200 * 64; i++)
		{
			if(MOTORFISH_SERIAL.available())
				break;
			ClosedLoop.stepBackward();
			pos--;
			delay(1);
			int32_t mPos = Encoder.motorPos();
			DEBUG_INFO("Pos:%8d - Error:%+03d\n", mPos, mPos - pos);
		}
	}


	// disable servo mode
	ClosedLoop.disable();

	MOTORFISH_SERIAL.read();	
}

void speedTest(void)
{
	MOTORFISH_SERIAL << "MotorFish Speed/Acceleration Test\n\n";
	MOTORFISH_SERIAL << "WARNING WARNING WARNING WARNING WARNING\n";
	MOTORFISH_SERIAL << "The motor will move VERY fast from half turn up to 2 full turns\n";
	MOTORFISH_SERIAL << "in order to calculate maximum speed and accelerations\n";
	MOTORFISH_SERIAL << "PLEASE CHECK YOUR MECHANICS BEFORE STARTING - TEST CAN DAMAGE IT\n";
	MOTORFISH_SERIAL << "Press <SPACE> to continue or any other key to abort...";
	while(!MOTORFISH_SERIAL.available())
		;
	if(MOTORFISH_SERIAL.read() != ' ')
		return;
	MOTORFISH_SERIAL << "\n";

	// enable servo mode
	ClosedLoop.enable();
	delay(100);

	int32_t pos = Encoder.motorPos();
	int32_t nPos = pos + 200 * 64;
	uint32_t tim = micros();
	uint32_t tOut = millis();
	ClosedLoop.setPosition(nPos);
	while(nPos - Encoder.motorPos() > 32 && millis() - tOut < 3000)
		;
	tim = micros() - tim;
	
	// give time to settle
	delay(300);
	
	// disable servo mode
	ClosedLoop.disable();
	
	if(millis() - tOut >= 3000)
		DEBUG_ERROR("Wooops... something went wrong\n");
	else
	{
		double speed = 200 * 64;
		speed *= 1e6;
		speed /= tim;
		DEBUG_INFO("Mean speed is %.0f steps/second\n", speed);
	}

	MOTORFISH_SERIAL << "Press any key to leave...";
	while(!MOTORFISH_SERIAL.available())
		;
	MOTORFISH_SERIAL.read();	
}

// restore settings to factory default
void restoreDefaultSettings()
{
	// erase the whole eeprom
	FishinoEEPROM.eraseAll();
	
	// initialize data
	FishinoEEPROM.write32u(MOTORFISH_EEPROM_MAGIC, MOTORFISH_EEPROM_MAGIC_VAL);
	FishinoEEPROM.write8u(MOTORFISH_MODE, 0x02);
	FishinoEEPROM.write8u(MOTORFISH_MOTOR_DIRECTION, 0x01);
	FishinoEEPROM.write8u(MOTORFISH_ENCODER_PLACEMENT, 0x01);

	FishinoEEPROM.write16s(MOTORFISH_KP, 10000);
	FishinoEEPROM.write16s(MOTORFISH_KI, 0);
	FishinoEEPROM.write16s(MOTORFISH_KD, 0);
	FishinoEEPROM.write16s(MOTORFISH_INTEGRALDAMP, 32767);
	FishinoEEPROM.write32u(MOTORFISH_SYNCTIME, 50);
	
	FishinoEEPROM.writeDouble(MOTORFISH_WINDING_CURRENT, 2.2);
}

// load configuration from eeprom
void loadConfig(void)
{
	// check if EEPROM has been initialized, otherwise
	// setup default settings
	if(FishinoEEPROM.read32u(MOTORFISH_EEPROM_MAGIC) != MOTORFISH_EEPROM_MAGIC_VAL)
		restoreDefaultSettings();

	// load current operating mode
	currentMode = FishinoEEPROM.read8u(MOTORFISH_MODE);
	if(currentMode > 2)
		currentMode = 0;

	// load motor direction
	uint8_t motorDirection = FishinoEEPROM.read8u(MOTORFISH_MOTOR_DIRECTION);
	Stepper.setDirection(motorDirection);
	Encoder.setOutputDirection(motorDirection);

	uint8_t encoderPlacement = FishinoEEPROM.read8u(MOTORFISH_ENCODER_PLACEMENT);
	Encoder.setPlacement(encoderPlacement);
	
	int16_t kp = FishinoEEPROM.read16s(MOTORFISH_KP);
	int16_t ki = FishinoEEPROM.read16s(MOTORFISH_KI);
	int16_t kd = FishinoEEPROM.read16s(MOTORFISH_KD);
	int16_t damp = FishinoEEPROM.read16s(MOTORFISH_INTEGRALDAMP);
	ClosedLoop.setPID(kp, ki, kd);
	ClosedLoop.setIntegralDamp(damp);
	
	double current = FishinoEEPROM.readDouble(MOTORFISH_WINDING_CURRENT);
	Stepper.setWindingCurrent(current);
	
	// start active mode
	setupMode(currentMode);
}


void showMainMenu(void)
{
	for(int i = 0; i < 40; i++)
		MOTORFISH_SERIAL << "\n";
	MOTORFISH_SERIAL << "MotorFish - MAIN MENU\n\n";
	MOTORFISH_SERIAL << "  1) Change running mode (";
	printMode(MOTORFISH_SERIAL, OperatingModes, currentMode);
	MOTORFISH_SERIAL << ")\n";

	MOTORFISH_SERIAL << "  2) Change encoder placement (";
	printMode(MOTORFISH_SERIAL, EncoderPlacements, Encoder.isBackPlacement() ? 0 : 1);
	MOTORFISH_SERIAL << ")\n";

	MOTORFISH_SERIAL << "  3) Change motor positive direction (";
	printMode(MOTORFISH_SERIAL, MotorDirections, Stepper.isDirectionCW() ? 0 : 1);
	MOTORFISH_SERIAL << ")\n";
	
	MOTORFISH_SERIAL << "  4) Enter calibration menu\n";
	MOTORFISH_SERIAL << "  5) Enter PID menu\n";
	MOTORFISH_SERIAL << "  6) Enter test menu\n";

	MOTORFISH_SERIAL << "\nSelect function:";

}

void calibrationMenu()
{
	while(true)
	{
		for(int i = 0; i < 40; i++)
			MOTORFISH_SERIAL << "\n";
		MOTORFISH_SERIAL << "MotorFish - CALIBRATION MENU\n\n";
		MOTORFISH_SERIAL << "  1) Run calibration (READ MANUAL FIRST!)\n";
		MOTORFISH_SERIAL << "  2) Dump calibration values\n";
		MOTORFISH_SERIAL << "  9) Erase calibration data\n";

		MOTORFISH_SERIAL << "\n  0) Return to main menu\n";

		MOTORFISH_SERIAL << "\nSelect function:";

		while(!MOTORFISH_SERIAL.available())
			;
		char c = MOTORFISH_SERIAL.read();
		MOTORFISH_SERIAL << c << "\n";
		switch(c)
		{
			case '1':
				calibrate();
				break;
				
			case '2':
				dumpCalibration();
				break;
				
			case '9':
				eraseCalibration();
				break;
				
			case '0':
			case 27:
				return;
				
			default:
				break;
		}
	}
}

void pidMenu()
{
	while(true)
	{
		for(int i = 0; i < 40; i++)
			MOTORFISH_SERIAL << "\n";
		MOTORFISH_SERIAL << "MotorFish - PID MENU\n\n";

		MOTORFISH_SERIAL << "  1) Change motor current (" << Stepper.getWindingCurrent() << ")\n";
		MOTORFISH_SERIAL << "  2) Change Kp value (" << ClosedLoop.getKp() << ")\n";
		MOTORFISH_SERIAL << "  3) Change Ki value (" << ClosedLoop.getKi() << ")\n";
		MOTORFISH_SERIAL << "  4) Change Kd value (" << ClosedLoop.getKd() << ")\n";
		MOTORFISH_SERIAL << "  5) Change integral damping factor (" << ClosedLoop.getIntegralDamp() << ")\n";

		MOTORFISH_SERIAL << "\n  0) Return to main menu\n";

		MOTORFISH_SERIAL << "\nSelect function:";

		while(!MOTORFISH_SERIAL.available())
			;
		char c = MOTORFISH_SERIAL.read();
		MOTORFISH_SERIAL << c << "\n";
		switch(c)
		{
			case '1':
			{
				double current = Stepper.getWindingCurrent();
				if(readDouble("Enter motor current", current))
				{
					Stepper.setWindingCurrent(current);
					FishinoEEPROM.writeDouble(MOTORFISH_WINDING_CURRENT, current);
				}
				break;
			}
			case '2':
			{
				int16_t kp = ClosedLoop.getKp();
				if(readInt16("Enter Kp value", kp))
				{
					ClosedLoop.setKp(kp);
					FishinoEEPROM.write16s(MOTORFISH_KP, kp);
				}
				break;
			}
			case '3':
			{
				int16_t ki = ClosedLoop.getKi();
				if(readInt16("Enter Ki value", ki))
				{
					ClosedLoop.setKi(ki);
					FishinoEEPROM.write16s(MOTORFISH_KI, ki);
				}
				break;
			}
			case '4':
			{
				int16_t kd = ClosedLoop.getKd();
				if(readInt16("Enter Kd value", kd))
				{
					ClosedLoop.setKd(kd);
					FishinoEEPROM.write16s(MOTORFISH_KD, kd);
				}
				break;
			}
			case '5':
			{
				int16_t damp = ClosedLoop.getIntegralDamp();
				if(readInt16("Enter integral damping factor", damp))
				{
					ClosedLoop.setIntegralDamp(damp);
					FishinoEEPROM.write16s(MOTORFISH_INTEGRALDAMP, damp);
				}
				break;
			}
			case '0':
			case 27:
				return;
				
			default:
				break;
		}
	}
}

void testMenu()
{
	while(true)
	{
		for(int i = 0; i < 40; i++)
			MOTORFISH_SERIAL << "\n";
		MOTORFISH_SERIAL << "MotorFish - TEST MENU\n\n";
		MOTORFISH_SERIAL << "  1) Encoder test\n";
		MOTORFISH_SERIAL << "  2) Open loop stepper test\n";
		MOTORFISH_SERIAL << "  3) Closed loop stepper test\n";
		MOTORFISH_SERIAL << "  4) Speed/acceleration test\n";

		MOTORFISH_SERIAL << "\n  0) Return to main menu\n";

		MOTORFISH_SERIAL << "\nSelect function:";

		while(!MOTORFISH_SERIAL.available())
			;
		char c = MOTORFISH_SERIAL.read();
		MOTORFISH_SERIAL << c << "\n";
		switch(c)
		{
			case '1':
				encoderTest();
				break;
				
			case '2':
				openLoopTest();
				break;
				
			case '3':
				closedLoopTest();
				break;
				
			case '4':
				speedTest();
				break;
				
			case '0':
			case 27:
				return;
				
			default:
				break;
		}
	}
}


// process main menu commands
// this is a non-blocking function
// it will block ONLY when entering a submenu
void processMainMenu(void)
{
	char c = MOTORFISH_SERIAL.read();
	MOTORFISH_SERIAL << c << "\n";
	
	switch(c)
	{
		case '1':
			currentMode++;
			currentMode %= 3;
			FishinoEEPROM.write8u(MOTORFISH_MODE, currentMode);
			setupMode(currentMode);
			break;
			
		case '2':
		{
			// disable motor before change settings!
			uint8_t prevMode = setupMode(0);
			
			if(Encoder.isBackPlacement())
			{
				Encoder.setFrontPlacement();
				FishinoEEPROM.write8u(MOTORFISH_ENCODER_PLACEMENT, 0x0);
			}
			else
			{
				Encoder.setBackPlacement();
				FishinoEEPROM.write8u(MOTORFISH_ENCODER_PLACEMENT, 0x1);
			}
			
			// re-enable
			setupMode(prevMode);
			
			break;
		}
			
		case '3':
		{
			// disable motor before change settings!
			uint8_t prevMode = setupMode(0);
			
			if(Stepper.isDirectionCW())
			{
				Stepper.setDirectionCCW();
				Encoder.setOutputDirectionCCW();
				FishinoEEPROM.write8u(MOTORFISH_MOTOR_DIRECTION, 0x0);
			}
			else
			{
				Stepper.setDirectionCW();
				Encoder.setOutputDirectionCW();
				FishinoEEPROM.write8u(MOTORFISH_MOTOR_DIRECTION, 0x1);
			}

			// re-enable
			setupMode(prevMode);
			
			break;
		}
			
		case '4':
		{
			uint8_t prevMode = setupMode(0);
			calibrationMenu();
			setupMode(prevMode);
			break;
		}
			
		case '5':
			pidMenu();
			break;

		case '6':
		{
			uint8_t prevMode = setupMode(0);
			testMenu();
			setupMode(prevMode);
			break;
		}
			
		default:
			break;
	}
	
	MOTORFISH_SERIAL << "\n\n";
	showMainMenu();
}

extern const uint32_t _flash_page_size;
extern const uint32_t _flash_row_size;

void setup()
{
	// initialize serial port
	MOTORFISH_SERIAL.begin(115200);
/*
	while(!MOTORFISH_SERIAL)
		;
*/
	DEBUG_SET_STREAM(MOTORFISH_SERIAL);

	// setup I/O 
	pinMode(STEP, INPUT_PULLDOWN);
	pinMode(DIR, INPUT_PULLDOWN);
	pinMode(ENABLE, INPUT_PULLDOWN);
	pinMode(YELLOW, OUTPUT);
	pinMode(BLUE, OUTPUT);
	pinMode(RED, OUTPUT);
	pinMode(WHITE, OUTPUT);
	
	// yellow is used for enable/disable display
	digitalWrite(YELLOW, LOW);
	
	// blue led is used for direction
	digitalWrite(BLUE, digitalRead(DIR));
	
	// red led is for fault condition
	digitalWrite(RED, LOW);

	// read config from eeprom
	loadConfig();
	
	// drop the menu
	showMainMenu();
}

void loopStepper(void)
{
	// keep enable pin watched
	bool dis = PORTD & _BV(0);
	bool enabled = Stepper.isEnergized();
	if((dis && enabled) || (!dis && !enabled))
	{
		enabled = !dis;

		// show enabled state with yellow led
		if(enabled)
			YELLOW_LAT |= _BV(YELLOW_BIT);
		else
			YELLOW_LAT &= ~_BV(YELLOW_BIT);
		
		// reset motor position and required position when enabling
		Stepper.energize(enabled);
	}
	
	// update direction led
	if(DIR_PORT & _BV(DIR_BIT))
		BLUE_LAT |= _BV(BLUE_BIT);
	else
		BLUE_LAT &= ~_BV(BLUE_BIT);
}

void loopServo(void)
{
	// keep enable pin watched
	bool enabled = ClosedLoop.isEnabled();
	bool dis = ENABLE_PORT & _BV(ENABLE_BIT);
	if((dis && enabled) || (!dis && !enabled))
	{
		enabled = !dis;
		ClosedLoop.enable(enabled);

		// show enabled state with yellow led
		if(enabled)
		{
			YELLOW_LAT |= _BV(YELLOW_BIT);
		}
		else
		{
			YELLOW_LAT &= ~_BV(YELLOW_BIT);
		}
	}
	
	// update direction led
	if(DIR_PORT & _BV(DIR_BIT))
		BLUE_LAT |= _BV(BLUE_BIT);
	else
		BLUE_LAT &= ~_BV(BLUE_BIT);
	
	// keep driver's error bit watched
	if(Stepper.isError())
	{
		RED_LAT |= _BV(RED_BIT);
	}
	else
	{
		RED_LAT &= ~_BV(RED_BIT);
	}
}

void loop()
{
	// check if we got a character from serial port
	if(MOTORFISH_SERIAL.available())
		processMainMenu();
	
	// process current mode, if any active
	if(currentMode == 1)
		loopStepper();
	else if(currentMode == 2)
		loopServo();
}
