//////////////////////////////////////////////////////////////////////////////////////
//						 		ClosedLoop.h										//
//																					//
//			Library to handle MotorFish closed loop stepper driver board			//
//																					//
//		Copyright (c) 2019 Massimo Del Fedele.  All rights reserved.				//
//																					//
//	Redistribution and use in source and binary forms, with or without				//
//	modification, are permitted provided that the following conditions are met:		//
//																					//
//	- Redistributions of source code must retain the above copyright notice,		//
//	  this list of conditions and the following disclaimer.							//
//	- Redistributions in binary form must reproduce the above copyright notice,		//
//	  this list of conditions and the following disclaimer in the documentation		//
//	  and/or other materials provided with the distribution.						//
//																					//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"		//
//	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE		//
//	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE		//
//	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE		//
//	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR				//
//	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF			//
//	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS		//
//	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN			//
//	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)			//
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE		//
//	POSSIBILITY OF SUCH DAMAGE.														//
//																					//
//  Version 7.8.0 -- March 2019		Initial version									//
//																					//
//////////////////////////////////////////////////////////////////////////////////////
#ifndef __CLOSEDLOOP_H
#define __CLOSEDLOOP_H

#include <Arduino.h>

class ClosedLoopClass
{
	private:
		
		// time between sync, in microseconds
		static volatile uint32_t _syncTime;
		
		// syncing active flag
		static volatile bool _syncing;
		
		// enabled flag
		static volatile bool _enabled;
		
		// the PID variables (-32768..+32767)
		static volatile int16_t _Kp, _Kd, _Ki;
		
		// integral factor damping
		// used as integralDamp / 32767 to reduce effects of farthest integral terms
		// should be near unity (32767)
		static volatile int32_t _integralDamp;

		// running pid variables
		static volatile int32_t _prevDiff;
		static volatile int32_t _integralTerm;

		// the required motor position
		static volatile int32_t _requiredPosition;
		
		// the sync timed routine
		static void _syncISR(void);
		
	protected:

	public:

		// constructor
		ClosedLoopClass();

		// destructor
		~ClosedLoopClass();
		
		// set PID parameters
		static void setPID(int16_t kp, int16_t ki, int16_t kd);
		
		// set individual PID parameters
		static void setKp(int16_t kp);
		static void setKi(int16_t ki);
		static void setKd(int16_t kd);

		// get PID parameters
		static int16_t getKp(void)
			{ return _Kp; }
		static int16_t getKi(void)
			{ return _Ki; }
		static int16_t getKd(void)
			{ return _Kd; }
		
		// set the integral damping factor
		static void setIntegralDamp(int32_t d);
		
		// get integral damping factor
		static int16_t getIntegralDamp(void)
			{ return _integralDamp; }
		
		// set sync time (microseconds)
		static void setSyncTime(uint32_t tim);
		
		// get sync time (microseconds)
		static uint32_t getSyncTime(void)
			{ return _syncTime; }
		
		// start syncing position
		static void startSync(void);
		
		// stop syncing position
		static void stopSync(void);
		
		// check if syncing
		static bool isSyncing(void)
			{ return _syncing; }
		
		// enable/disable the motor
		static void enable(bool en = true);
		static void disable(void)
			{ enable(false); }
			
		// check if enabled
		static bool isEnabled(void)
			{ return _enabled; }
			
		// execute a step
		static void step(bool dir);
		static void stepForward(void)
			{ step(true); }
		static void stepBackward(void)
			{ step(false); }
			
		// get positions
		int32_t requiredPosition(void)
			{ return _requiredPosition; }
		int32_t encoderPosition(void)
			{ return Encoder.motorPos(); }
			
		// set position
		// WARNING - TESTIN ONLY
		// the position wil be reached at MAXIMUM POSSIBLE SPEED
		void setPosition(int32_t nPos)
			{ _requiredPosition = nPos; }

};

#define ClosedLoop __getClosedLoopClass()

extern ClosedLoopClass &__getClosedLoopClass(void);

#endif
