//////////////////////////////////////////////////////////////////////////////////////
//						 		Encoder.h											//
//																					//
//			Library to handle MotorFish closed loop stepper driver board			//
//																					//
//		Copyright (c) 2019 Massimo Del Fedele.  All rights reserved.				//
//																					//
//	Redistribution and use in source and binary forms, with or without				//
//	modification, are permitted provided that the following conditions are met:		//
//																					//
//	- Redistributions of source code must retain the above copyright notice,		//
//	  this list of conditions and the following disclaimer.							//
//	- Redistributions in binary form must reproduce the above copyright notice,		//
//	  this list of conditions and the following disclaimer in the documentation		//
//	  and/or other materials provided with the distribution.						//
//																					//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"		//
//	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE		//
//	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE		//
//	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE		//
//	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR				//
//	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF			//
//	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS		//
//	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN			//
//	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)			//
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE		//
//	POSSIBILITY OF SUCH DAMAGE.														//
//																					//
//  Version 7.8.0 -- March 2019		Initial version									//
//																					//
//////////////////////////////////////////////////////////////////////////////////////
#ifndef __ENCODER_H
#define __ENCODER_H

#include <Arduino.h>

#include <SPI.h>

// convert position from encoder to microstepping
// valid for 64 microsteps/step and 12 bit encoder resolution
#define ENC2STEP(x) ((25 * x + 4) / 8)

// convert position from microstepping to encoder
// valid for 64 microsteps/step and 12 bit encoder resolution
#define STEP2ENC(x) ((16 * x + 25) / 50)

// define following line to use A/B/RI quadrature signals
// for encoder position reading -- interrupt driven
// if undefined, the SSI serial position is used
// quadrature signals are faster but behaves erratically with
// the board too near to stepper motor -- still no solution found
#define USE_QUADRATURE_SIGNALS

// define following line to use fast-SSI reading without the need
// to wait monoflop timeout. By now no problem using this mode
#define USE_FAST_SSI

enum class EncoderErrors
{
	SSI_NOT_READY = 0x01,	// SSI interface's monoflop still busy
	SSI_BAD_VALUE = 0x02,	// got a value over 4095 from SSI
	ERROR_TOO_BIG = 0x04,	// difference between quadrature data and SSI too big
};

class EncoderClass
{
	private:
		
		// as it's not safe to print errors from inside ISR, we use
		// some flags that can be queried by main application
		volatile static uint16_t _errorFlags;
		volatile static uint16_t _errorSSI;
		volatile static uint32_t _errorPrevMPos, _errorCurMPos;
		volatile static uint16_t _errorMPOS_QUAD, _error_MPOS_SSI;

#ifdef USE_QUADRATURE_SIGNALS
		static void _quadratureISR(void) __attribute__((interrupt(), nomips16));

		// quadrature reading value
		volatile static uint8_t _quadratureData;

#endif

		// the encoder placement (true = backside, false = front side)
		volatile static bool _placement;
		
		// the encoder positive direction (true = CW, false = CCW)
		// this is stored to allow encoder placement change
		volatile static bool _direction;
		
		// enable/disable calibration
		volatile static bool _calibrationEnabled;

		// the absolute motor position
		volatile static int32_t _motorPos;
		
		// last really read SSI position -- returned when
		// SSI interface is busy instead of reading the fresh one
		volatile static uint16_t _lastReadSSIPos;
		
		// previous SSI reading used to keep motorPos updated
		volatile static int32_t _prevSSIPos;

		// SSI reading for motor 0 position
		volatile static uint16_t _zeroPos;
		
		// calibration data
		static bool _hasCalibrationData;
		static int8_t *_calibrationData;
		
		// eeprom mode for I2C
		static bool _eepromMode;
		
		// read encoder register
		static uint16_t readReg(uint8_t reg);
		
		// write encoder register
		static void writeReg(uint8_t reg, uint16_t data);
		
		// SPI settings used to read SSI position
#ifndef USE_FAST_SSI
		static SPISettings _spiSettings;
#endif
		
		// reset motor position
		// (set to 0, re-read zero SSI position)
		static void _resetMotorPos(void);

		// reads absolute position from encoder
		// a 12 bit number from 0 to 4095
		// this is a RAW read, without any calibration data applied
		static uint16_t _readSSIPosition(void);
		
	protected:

	public:
		
		// power down rates
		enum PDN_RATES
		{
			PDN_1_128	= 0x00,
			PDN_1_256	= 0x01,
			PDN_1_512	= 0x02,
			PDN_1_1024	= 0x03
		};
		
		// regulator voltages
		enum REG_VOLTAGES
		{
			RV_3		= 0x00,
			RV_3_3		= 0x01
		};
		
		// UVW output selection
		enum UVW_SEL
		{
			UVW_TACHO	= 0x00,
			SINE_DIFF	= 0x01
		};
		
		// error pin output
		enum ERR_PIN_SEL
		{
			ERR_SIGNAL		= 0x00,
			ERR_AMPLITUDE	= 0x01
		};
		
		// output direction
		enum OUTPUT_DIR
		{
			DIR_POS		= 0x00,
			DIR_NEG		= 0x01
		};
		
		// output position type
		enum OUTPUT_POS_TYPE
		{
			POS_RELATIVE	= 0x00,
			POS_ABSOLUTE	= 0x01
		};
		
		// VOUT/TOUT selection
		enum VOUT_TOUT_SEL
		{
			VOUT_TOUT_POS	= 0x00,
			VOUT_TOUT_TACHO	= 0x01
		};
		
		// linear voltage period
		enum LINEAR_VOLTAGE_PERIOD
		{
			PERIOD_360	= 0x00,
			PERIOD_180	= 0x01,
			PERIOD_90	= 0x10,
			PERIOD_45	= 0x11
		};
		
		// SSI output setting
		enum SSI_SETTINGS
		{
			SSI_NO_RING			= 0x00,
			SSI_RING_NOREFRESH	= 0x01,
			SSI_RING_REFRESH	= 0x03
		};
		
		// tacho ranges
		enum TACHO_RANGES
		{
			TACHO_122880	= 0x00,
			TACHO_61440		= 0x01,
			TACHO_30720		= 0x02,
			TACHO_15360		= 0x03,
			TACHO_7680		= 0x04,
			TACHO_3840		= 0x05,
			TACHO_1920		= 0x06,
			TACHO_1960		= 0x07
		};
		
		// UVW periods/turn
		enum UVW_PERIODS
		{
			UVW_1		= 0x00,
			UVW_2		= 0x01,
			UVW_3		= 0x02,
			UVW_4		= 0x03,
			UVW_5		= 0x04,
			UVW_6		= 0x05,
			UVW_7		= 0x06,
			UVW_8		= 0x07
		};
		
		// interpolation factor rate
		enum INTERP_RATE
		{
			RATE_4096	= 0x00,
			RATE_2048	= 0x01,
			RATE_1024	= 0x02,
			RATE_512	= 0x03,
			RATE_256	= 0x04,
			RATE_128	= 0x05,
			RATE_64		= 0x06,
			RATE_32		= 0x07
		};

		// constructor
		EncoderClass();

		// destructor
		~EncoderClass();
		
		// output direction - encoder placement INDEPENDENT
		static void setOutputDirection0(OUTPUT_DIR dir);
		static OUTPUT_DIR getOutputDirection0(void);
		
		// set encoder placement
		static void setBackPlacement(void);
		static void setFrontPlacement(void);
		static void setPlacement(bool p);
		
		// get encoder placement
		static bool isBackPlacement(void)
			{ return _placement; }
			
		static bool isFrontPlacement(void)
			{ return !_placement; }
			
		static bool getPlacement(void)
			{ return _placement; }
			
		////////////////////////////////////
		// set/get some encoder registers
		////////////////////////////////////
		
		// set clear eeprom mode for registers
		static void setEEPROMMode(void)
			{ _eepromMode = true; }
		static void setRAMMode(void)
			{ _eepromMode = false; }
		
		// interpolator power
		static void enableInterpolator(bool en = true);
		static void disableInterpolator(void)
			{ enableInterpolator(false); }
		static bool isInterpolatorEnabled(void);
		
		// AGC
		static void enableAGC(bool en = true);
		static void disableAGC(void)
			{ enableAGC(false); }
		static bool isAGCEnabled(void);
		
		// interpolator delay
		static void enableInterpolatorDelay(bool en = true);
		static void disableInterpolatorDelay(void)
			{ enableInterpolatorDelay(false); }
		static bool isInterpolatorDelayEnabled(void);
		
		// power down rate
		static void setPowerDownRate(PDN_RATES rate);
		static PDN_RATES getPowerDownRate(void);
		
		// power down state
		static void enablePowerDown(bool en = true);
		static void disablePowerDown(void)
			{ enablePowerDown(false); }
		static bool isPowerDownEnabled(void);
		
		// regulator voltage
		static void setRegulatorVoltage(REG_VOLTAGES v);
		static REG_VOLTAGES getRegulatorVoltage(void);
		
		// device address
		static void setDeviceAddress(uint8_t addr);
		static uint8_t getDeviceAddress(void);
		
		// A/B/Ri outputs
		static void enableABRIOutputs(bool en = true);
		static void disableABRIOutputs(void)
			{ enableABRIOutputs(false); }
		static bool isABRIOutputsEnabled(void);
		
		// UVW output type
		static void setUVWOutputType(UVW_SEL sel);
		static UVW_SEL getUVWOutputType(void);
		
		// error output type
		void setErrorOutputType(ERR_PIN_SEL sel);
		ERR_PIN_SEL getErrorOutputType(void);
		
		// output direction - encoder placement dependent
		// true is CW
		static void setOutputDirectionCW(void);
		static void setOutputDirectionCCW(void);
		static void setOutputDirection(bool dir);
		static bool isOutputDirectionCW(void)
			{ return _direction; }
		static bool isOutputDirectionCCW(void)
			{ return !_direction; }
		static bool getOutputDirection(void)
			{ return _direction; }
		
		// zero position
		static void setZeroPosition(uint16_t pos);
		static uint16_t getZeroPosition(void);
		
		// output position type
		static void setOutputPositionType(OUTPUT_POS_TYPE type);
		static OUTPUT_POS_TYPE getOutputPositionType(void);
		
		// digital hysteresis
		static void setDigitalHysteresis(uint8_t h);
		static uint8_t getDigitalHysteresis(void);
		
		// Vout/Tout ooutput type
		static void setVoutToutOutputType(VOUT_TOUT_SEL sel);
		static VOUT_TOUT_SEL getVoutToutOutputType(void);
		
		// linear voltage period
		static void setLinearVoltagePeriod(LINEAR_VOLTAGE_PERIOD period);
		static LINEAR_VOLTAGE_PERIOD getLinearVoltagePeriod(void);
		
		// SSI output setting
		static void setSSISettings(SSI_SETTINGS s);
		static SSI_SETTINGS getSSISettings(void);
		
		// Tacho range
		static void setTachoRange(TACHO_RANGES r);
		static TACHO_RANGES getTachoRange(void);

		// UVW period
		static void setUVWPeriod(UVW_PERIODS period);
		static UVW_PERIODS getUVWPeriod(void);
		
		// interpolator factor rate
		static void setInterpolatorRate(INTERP_RATE rate);
		static INTERP_RATE getInterpolatorRate(void);
		
		// check if position is valid
		static bool isRelativePositionValid(void);
		static bool isAbsolutePositionValid(void);
		
		// read absolute and relative position
		static uint16_t readRelativePosition(void);
		static uint16_t readAbsolutePosition(void);
		
		// check magnet status
		static bool isMagnetTooFar(void);
		static bool isMagnetTooClose(void);
		static bool isMagnetOk(void);
			
		// tacho
		static bool isTachoOverflow(void);
		static uint16_t readTacho(void);
		
		// read/write eeprom (8 bytes)
		static void writeEEProm(uint8_t reg, uint8_t data);
		static uint8_t readEEProm(uint8_t reg);
		
		// read SSI absolute position with calibration data applied
		// this one ensure that NO CONCURRENT hardware read is made
		// if it can't read an updated position, just use previous one
		static uint16_t readSSIPosition();
		
		// get motor position (in stepper coordinates) - calibration data applied
		static int32_t motorPos();
			
		// reset motor position
		// (set to 0, re-read zero SSI position)
		static void resetMotorPos(void)
			{ _resetMotorPos(); }
			
		// enable/disable calibrated mode
		static void enableCalibration(bool en = true);
		static void disableCalibration(void)
			{ enableCalibration(false); }
			
		// dump calibration data on stream
		static void dumpCalibration(void);
		
		// read error flags and various related data
		static uint16_t errorFlags(void);
		static void clearErrors(void);
		static void printErrors(void);
};

#define Encoder __getEncoderClass()

extern EncoderClass &__getEncoderClass(void);

#endif
