//////////////////////////////////////////////////////////////////////////////////////
//																					//
//								FishinoClient.cpp									//
//						Library for ESP8266 WiFi module								//
//					Created by Massimo Del Fedele, 2015								//
//																					//
//  Copyright (c) 2015, 2016 and 2017 Massimo Del Fedele.  All rights reserved.		//
//																					//
//	Redistribution and use in source and binary forms, with or without				//
//	modification, are permitted provided that the following conditions are met:		//
//																					//
//	- Redistributions of source code must retain the above copyright notice,		//
//	  this list of conditions and the following disclaimer.							//
//	- Redistributions in binary form must reproduce the above copyright notice,		//
//	  this list of conditions and the following disclaimer in the documentation		//
//	  and/or other materials provided with the distribution.						//
//																					//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"		//
//	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE		//
//	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE		//
//	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE		//
//	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR				//
//	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF			//
//	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS		//
//	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN			//
//	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)			//
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE		//
//	POSSIBILITY OF SUCH DAMAGE.														//
//																					//
//	VERSION 1.0.0 - INITIAL VERSION													//
//	VERSION 2.0.0 - 06/01/2016 - REWROTE SPI INTERFACE AND ERROR HANDLING			//
//	VERSION 4.0.0 - 01/01/2017 - REWROTE SPI INTERFACE AND ERROR HANDLING			//
//	VERSION 5.1.0 - 04/05/2017 - USE NEW DEBUG LIBRARY								//
//	VERSION 5.2.0 - 20/05/2017 - USE NEW DEBUG LIBRARY								//
//	Version 6.0.0 -- June 2017 - USE NEW DEBUG LIBRARY								//
//	Version 7.0.0 -- June 2017 - REWROTE SPI INTERFACE								//
//	Version 7.0.1 -- June 2017 - FIXED BUG ON MEGA									//
//	Version 7.1.0 - 20/10/2017 - AVOID AUTOMATIC SOCKET CLOSE FOR DELETED CLIENTS	//
//	Version 7.3.0 - 17/12/2017 - ADD WIFI MODULE DEEP SLEEP COMMAND					//
//	Version 7.3.2 - 05/01/2018 - FIXED RSSI AND MAC HANDLING/DISPLAY				//
//	Version 8.0.0 - 25/04/2020 - UPDATED FOR FIRMWARE 8.0.0							//
//																					//
//////////////////////////////////////////////////////////////////////////////////////

#define FISHINO_MODULE "FishinoClient"
#include "Fishino.h"

//uint16_t EthernetClient::_srcport = 49152;      //Use IANA recommended ephemeral port range 49152-65535

void FishinoClient::init(uint8_t sock)
{
	// create only if not already done
	if(ref)
		return;
	
	// create the socket
	if(sock == 0xff)
		sock = Fishino.createTcpSocket(ssl);
	
	// create reference object
	if(sock != 0xff)
		ref = new ReadBuf<uint8_t, FishinoClient>(sock);
	else
		ref = NULL;
}

// socket constructor
FishinoClient::FishinoClient(uint8_t sock)
{
	ssl = false;
	ref = NULL;
	init(sock);
}

// constructor
FishinoClient::FishinoClient()
{
	ssl = false;
	ref = NULL;
}

// copy constructor
FishinoClient::FishinoClient(FishinoClient const &c)
{
	ref = c.ref;
	if(ref)
		ref->addRef();
	ssl = c.ssl;
}

// destructor
FishinoClient::~FishinoClient()
{
	if(ref)
		ref->remRef();
}

// copy operator
FishinoClient const &FishinoClient::operator=(FishinoClient const &c)
{
	if(ref)
		ref->remRef();
	ref = c.ref;
	if(ref)
		ref->addRef();
	ssl = c.ssl;
	return *this;
}

// attach to an already opene socket
void FishinoClient::attach(uint8_t sock)
{
	// disconnect from any previous socket
	if(ref)
		ref->remRef();
	ref = NULL;
	
	// create reference object
	init(sock);
	
	return;
}

int FishinoClient::connect(const char* host, uint16_t port)
{
	// init, if needed
	init(0xff);
	
	// connect
	return Fishino.connect(ref->getSocket(), host, port);
}

int FishinoClient::connect(IPAddress ip, uint16_t port)
{
	// init, if needed
	init(0xff);
	
	return Fishino.connect(ref->getSocket(), ip, port);
}

size_t FishinoClient::write(uint8_t b)
{
	// init, if needed
	init(0xff);
	
	return write(&b, 1);
}

size_t FishinoClient::write(const uint8_t *buf, size_t size)
{
	// init, if needed
	init(0xff);
	
	bool res = Fishino.write(ref->getSocket(), buf, size);
	if(res)
		return size;

	setWriteError();
	return 0;
}

int FishinoClient::available()
{
	// init, if needed
	init(0xff);
	
	return ref->avail();
}

int FishinoClient::read()
{
	// init, if needed
	init(0xff);
	
	uint8_t b;
	if(ref->read(&b, 1) == 1)
		return b;
	return -1;
}

int FishinoClient::read(uint8_t *buf, size_t size)
{
	// init, if needed
	init(0xff);
	
	return ref->read(buf, size);
}

int FishinoClient::peek()
{
	// init, if needed
	init(0xff);
	
	uint8_t b;
	if(ref->peek(b))
		return b;
	return -1;
}

void FishinoClient::flush()
{
	// init, if needed
	init(0xff);
	
	Fishino.flush(ref->getSocket());
}

void FishinoClient::stop()
{
	// init, if needed
	init(0xff);
	
	Fishino.disconnect(ref->getSocket());
}

uint8_t FishinoClient::connected()
{
	// init, if needed
	init(0xff);
	
	return Fishino.connected(ref->getSocket()) || ref->avail();
}

bool FishinoClient::status()
{
	// init, if needed
	init(0xff);
	
	return (Fishino.status(ref->getSocket()) != CLOSED) || ref->avail();
}

// the next function allows us to use the client returned by
// EthernetServer::available() as the condition in an if-statement.

FishinoClient::operator bool()
{
	return status();
}

// set buffered mode for tcp client
bool FishinoClient::setBufferedMode(bool b)
{
	// init, if needed
	init(0xff);
	
	return Fishino.setBufferedMode(ref->getSocket(), b);
}

// get buffered mode for tcp client
bool FishinoClient::getBufferedMode(void)
{
	// init, if needed
	init(0xff);
	
	return Fishino.getBufferedMode(ref->getSocket());
}

// disable/enable nagle for tcp client
bool FishinoClient::setNoDelay(bool b)
{
	// init, if needed
	init(0xff);
	
	return Fishino.setNoDelay(ref->getSocket(), b);
}

// get nagle status for tcp client
bool FishinoClient::getNoDelay(void)
{
	// init, if needed
	init(0xff);
	
	return Fishino.getNoDelay(ref->getSocket());
}

// set inactivity timeout for tcp client
bool FishinoClient::setForceCloseTime(uint32_t tim)
{
	// init, if needed
	init(0xff);
	
	return Fishino.setClientForceCloseTime(ref->getSocket(), tim);
}

// get inactivity timeout for tcp client
uint32_t FishinoClient::getForceCloseTime(void)
{
	// init, if needed
	init(0xff);
	
	return Fishino.getClientForceCloseTime(ref->getSocket());
}

bool FishinoClient::operator==(const FishinoClient& rhs)
{
	return ref == rhs.ref;
}

#ifndef _FISHINO_PIC32_
size_t FishinoClient::print(const __FlashStringHelper *s)
{
	// init, if needed
	init(0xff);
	
	if(Fishino.write(ref->getSocket(), s))
		return strlen_P(reinterpret_cast<PGM_P>(s));
	return 0;
}

size_t FishinoClient::println(const __FlashStringHelper *s)
{
	size_t n =	print(s);
	n += print(F("\r\n"));
	return n;
}
#endif

FishinoSecureClient::FishinoSecureClient(uint8_t sock)
{
	ssl = true;
	init(sock);
}

FishinoSecureClient::FishinoSecureClient()
{
	ssl = true;
}

// verify secure socket domain fingerprint
bool FishinoSecureClient::verifyFingerPrint(const char *domain, const char *fingerPrint)
{
	// init, if needed
	init(0xff);
	
	return Fishino.verifyFingerPrint(ref->getSocket(), domain, fingerPrint);
}

bool FishinoSecureClient::verifyFingerPrint(const __FlashStringHelper *domain, const __FlashStringHelper *fingerPrint)
{
	// init, if needed
	init(0xff);
	
	return Fishino.verifyFingerPrint(ref->getSocket(), domain, fingerPrint);
}

// set client certificate
bool FishinoSecureClient::setClientCertificate(const uint8_t *buf, uint16_t len)
{
	return Fishino.setClientCertificate(buf, len);
}

bool FishinoSecureClient::setClientCertificate()
{
	return Fishino.setClientCertificate();
}

// det client private key
bool FishinoSecureClient::setClientPrivateKey(const uint8_t *buf, uint16_t len)
{
	return Fishino.setClientPrivateKey(buf, len);
}

bool FishinoSecureClient::setClientPrivateKey()
{
	return Fishino.setClientPrivateKey();
}

// Don't validate the chain, just accept whatever is given.  VERY INSECURE!
void FishinoSecureClient::setInsecure()
{
	// init, if needed
	init(0xff);
	
	Fishino.setInsecure(ref->getSocket());
}

// Only check SHA1 fingerprint of certificate
bool FishinoSecureClient::setFingerprint(const uint8_t fingerprint[20])
{
	// init, if needed
	init(0xff);
	
	return Fishino.setFingerprint(ref->getSocket(), fingerprint);
}

bool FishinoSecureClient::setFingerprint(const char *fpStr)
{
	// init, if needed
	init(0xff);
	
	return Fishino.setFingerprint(ref->getSocket(), fpStr);
}

