/*
This example will receive multiple universes via Artnet and control a strip of ws2811 leds via
Adafruit's NeoPixel library: https://github.com/adafruit/Adafruit_NeoPixel
This example may be copied under the terms of the MIT license, see the LICENSE file for details

29/08/2019 - Modified by Emanuele Signoretta to work on Fishino boards.

*/


#include<FishinoArtnet.h>
#include <Fishino.h>
#include <FishinoUdp.h>
#include <Adafruit_NeoPixel.h>
#include <SPI.h>


//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////
// CONFIGURATION DATA    -- ADAPT TO YOUR NETWORK !!!
// DATI DI CONFIGURAZIONE -- ADATTARE ALLA PROPRIA RETE WiFi !!!
#ifndef __MY_NETWORK_H

// here pur SSID of your network
// inserire qui lo SSID della rete WiFi
#define MY_SSID ""

// here put PASSWORD of your network. Use "" if none
// inserire qui la PASSWORD della rete WiFi -- Usare "" se la rete non ￨ protetta
#define MY_PASS ""

// here put required IP address (and maybe gateway and netmask!) of your Fishino
// comment out this lines if you want AUTO IP (dhcp)
// NOTE : if you use auto IP you must find it somehow !
// inserire qui l'IP desiderato ed eventualmente gateway e netmask per il fishino
// commentare le linee sotto se si vuole l'IP automatico
// nota : se si utilizza l'IP automatico, occorre un metodo per trovarlo !
#define IPADDR  192, 168,   1, 255
#define GATEWAY 192, 168,   1, 1
#define NETMASK 255, 255, 255, 0

#endif
//                    END OF CONFIGURATION DATA                      //
//                       FINE CONFIGURAZIONE                         //
///////////////////////////////////////////////////////////////////////

// define ip address if required
// NOTE : if your network is not of type 255.255.255.0 or your gateway is not xx.xx.xx.1
// you should set also both netmask and gateway
#ifdef IPADDR
IPAddress Ip(IPADDR);
#ifdef GATEWAY
IPAddress gw(GATEWAY);
#else
IPAddress gw(ip[0], ip[1], ip[2], 1);
#endif
#ifdef NETMASK
IPAddress nm(NETMASK);
#else
IPAddress nm(255, 255, 255, 0);
#endif
#endif


// Neopixel settings
const int numLeds = 270; // change for your setup
const int numberOfChannels = numLeds * 3; // Total number of channels you want to receive (1 led = 3 channels)
const byte dataPin = 2;
Adafruit_NeoPixel leds = Adafruit_NeoPixel(numLeds, dataPin, NEO_GRB + NEO_KHZ800);

// Artnet settings
FishinoArtnet artnet;
const int startUniverse = 0; // CHANGE FOR YOUR SETUP most software this is 1, some software send out artnet first universe as 0.

// Check if we got all universes
const int maxUniverses = numberOfChannels / 512 + ((numberOfChannels % 512) ? 1 : 0);
bool universesReceived[maxUniverses];
bool sendFrame = 1;
int previousDataLength = 0;

// connect to wifi – returns true if successful or false if not
boolean ConnectWifi(void) {
	boolean state = true;
	// reset and test WiFi module
	// resetta e testa il modulo WiFi
	while (!Fishino.reset())
		Serial << F("Fishino RESET FAILED, RETRYING...\n");
	Serial << F("Fishino WiFi RESET OK\n");

	Fishino.setPhyMode(PHY_MODE_11N);

	// go into station mode
	// imposta la modalità stazione
	Fishino.setMode(STATION_MODE);

	// try forever to connect to AP
	// tenta la connessione finchè non riesce
	Serial << F("Connecting to AP...");
	while (!Fishino.begin(MY_SSID, MY_PASS)) {
		Serial << ".";
		delay(2000);
	}
	Serial << "OK\n";


	// setup IP or start DHCP client
	// imposta l'IP statico oppure avvia il client DHCP
#ifdef IPADDR
	Fishino.config(Ip, gw, nm);
#else
	Fishino.staStartDHCP();
#endif

	// wait till connection is established
	Serial << F("Waiting for IP...");
	while (Fishino.status() != STATION_GOT_IP) {
		Serial << ".";
		delay(500);
	}
	Serial << "OK\n";

	return state;
}

void initTest() {
	for (int i = 0 ; i < numLeds ; i++)
		leds.setPixelColor(i, 127, 0, 0);
	leds.show();
	delay(500);
	for (int i = 0 ; i < numLeds ; i++)
		leds.setPixelColor(i, 0, 127, 0);
	leds.show();
	delay(500);
	for (int i = 0 ; i < numLeds ; i++)
		leds.setPixelColor(i, 0, 0, 127);
	leds.show();
	delay(500);
	for (int i = 0 ; i < numLeds ; i++)
		leds.setPixelColor(i, 0, 0, 0);
	leds.show();
}

void onDmxFrame(uint16_t universe, uint16_t length, uint8_t sequence, uint8_t* data) {


	sendFrame = 1;
	// set brightness of the whole strip
	if (universe == 15) {
		leds.setBrightness(data[0]);
		leds.show();
	}

	// Store which universe has got in
	if ((universe - startUniverse) < maxUniverses)
		universesReceived[universe - startUniverse] = 1;

	for (int i = 0 ; i < maxUniverses ; i++) {
		if (universesReceived[i] == 0) {
			//Serial.println("Broke");
			sendFrame = 0;
			break;
		}
	}
//////////////////////////////Modificare qui
	// read universe and put into the right part of the display buffer
	for (int i = 0; i < length / 3; i++) {
		int led = i + (universe - startUniverse) * (previousDataLength / 3);
		if (led < numLeds)
			leds.setPixelColor(led, data[i * 3], data[i * 3 + 1], data[i * 3 + 2]);
	}
	previousDataLength = length;

	if (sendFrame) {
		leds.show();
		// Reset universeReceived to 0
		memset(universesReceived, 0, maxUniverses);
	}
}

void setup() {
	Serial.begin(115200);
	ConnectWifi();
	artnet.begin();
	leds.begin();
	initTest();

	// this will be called for each packet received
	artnet.setArtDmxCallback(onDmxFrame);
}

void loop() {
	// we call the read function inside the loop
	artnet.read();
}
