#include "FishinoEEPROM.h"

#include "utils/flash.h"

// those are addresses, NOT variables nor constants
// they must be used with &operator
extern const uint8_t _eeprom_start;
extern const uint8_t _eeprom_size;
extern const uint8_t _flash_page_size;
extern const uint8_t _flash_row_size;

const uint32_t	EEPROM_START	= (uint32_t)&_eeprom_start;
const size_t	EEPROM_SIZE		= (size_t)(&_eeprom_size);
const size_t	FLASH_PAGE_SIZE	= (size_t)(&_flash_page_size);
const size_t	FLASH_ROW_SIZE	= (size_t)(&_flash_row_size);

const uint32_t	FLASH_PAGE_MASK	= FLASH_PAGE_SIZE - 1;
const uint32_t	FLASH_ROW_MASK	= FLASH_ROW_SIZE - 1;

// write a full eeprom page
// NOTE: addr must be page-aligned and buf must contain the whole page data
bool FishinoEEPROMClass::writePage(uint32_t addr, void const *buf)
{
	// align address to page boundary and convert it to flash address
	addr = EEPROM_START + (addr & ~FLASH_PAGE_MASK);

	// first check if page can be written without erasing
	uint32_t const *bufP = (uint32_t const *)buf;
	uint32_t const *flashP = (uint32_t *)addr;
	bool erase = false;
	for(size_t i = 0; i < FLASH_PAGE_SIZE / sizeof(uint32_t); i+= sizeof(uint32_t))
	{
		if((*bufP & *flashP) != *bufP)
		{
			erase = true;
			break;
		}
		bufP++;
		flashP++;
	}
	if(erase)
		if(eraseFlashPage((void *)addr))
			return false;
	return writeFlashPage((void *)addr, buf) == 0;
}

// get total EEPROM size, in bytes
size_t FishinoEEPROMClass::size(void)
{
	return EEPROM_SIZE;
}

// get page size in bytes
size_t FishinoEEPROMClass::pageSize(void)
{
	return FLASH_PAGE_SIZE;
}

// get row size
size_t FishinoEEPROMClass::rowSize(void)
{
	return FLASH_ROW_SIZE;
}

// erase all EEPROM area
bool FishinoEEPROMClass::eraseAll(void)
{
	size_t nPages = EEPROM_SIZE / FLASH_PAGE_SIZE;
	int32_t page = EEPROM_START;
	for(size_t i = 0; i < nPages; i++)
	{
		if(eraseFlashPage((void *)page))
			return false;
		page += FLASH_PAGE_SIZE;
	}
	return true;
}

// erase an eeprom page
// note : addr must be page-aligned
bool FishinoEEPROMClass::erasePage(uint32_t addr)
{
	// align address to page boundary and convert it to flash address
	addr = EEPROM_START + (addr & ~FLASH_PAGE_MASK);

	return eraseFlashPage((void *)addr) == 0;
}

// write data to eeprom
// note : addr must be 32-bit aligned
bool FishinoEEPROMClass::write32u(uint32_t addr, uint32_t data)
{
	// align address to 32 bit and convert to flash address
	addr = EEPROM_START + (addr & 0xfffffffc);
	
	// read existing word
	uint32_t prevData = *(uint32_t *)addr;
	
	// if data can be written directly, just do it
	if((prevData & data) == data)
		return writeFlashWord((void *)addr, data) == 0;
	
	// nope, we shall erase the page first
	// so store it inside a ram buffer, erase it, replace the word
	// and rewrite to flash
	uint8_t *buf = (uint8_t *)malloc(FLASH_PAGE_SIZE);
	uint32_t pageAddr = addr & ~FLASH_PAGE_MASK;
	memcpy(buf, (void *)pageAddr, FLASH_PAGE_SIZE);
	buf[addr - pageAddr] = data;
	uint32_t res = writeFlashPage((void *)pageAddr, buf);
	free(buf);
	return res == 0;
}

// note : addr must be 16 bit aligned
bool FishinoEEPROMClass::write16u(uint32_t addr, uint16_t data)
{
	// align address to 16 bit
	addr &= 0xfffffffe;
	
	// get flash address
	uint32_t flashAddr = EEPROM_START + (addr &0xfffffffc);
	
	// read existing word
	uint32_t prevData = *(uint32_t *)(flashAddr);
	
	// replace correct half word
	if(addr & 0x02)
		// upper half
		prevData = (prevData & 0x0000ffff) | (((uint32_t)data) << 16);
	else
		// lower half
		prevData = (prevData & 0xffff0000) | data;
	return write32u(addr & 0xfffffffc, prevData);
}

bool FishinoEEPROMClass::write8u(uint32_t addr, uint8_t data)
{
	// get flash address
	uint32_t flashAddr =  EEPROM_START + (addr &0xfffffffc);
	
	// read existing word
	uint32_t prevData = *(uint32_t *)(flashAddr);
	
	// replace correct byte
	switch(addr & 0x03)
	{
		case 0:
			prevData = (prevData & 0xffffff00) | data;
			break;
		case 1:
			prevData = (prevData & 0xffff00ff) | (((uint16_t)data) << 8);
			break;
		case 2:
			prevData = (prevData & 0xff00ffff) | (((uint32_t)data) << 16);
			break;
		case 3:
			prevData = (prevData & 0x00ffffff) | (((uint32_t)data) << 24);
			break;
	}
	return write32u(addr & 0xfffffffc, prevData);
}

bool FishinoEEPROMClass::write(uint32_t addr, uint8_t const *buf, size_t len)
{
	// get address ofset from its page
	uint32_t addrInPage = addr & FLASH_PAGE_MASK;

	// if the address is not on a page boundary, write first part to
	// move to a page boundary
	if(addrInPage)
	{
		uint32_t pageAddr = addr & ~FLASH_PAGE_MASK;
		uint8_t *pBuf = (uint8_t *)malloc(FLASH_PAGE_SIZE);
		memcpy(pBuf, (void *)(EEPROM_START + pageAddr), FLASH_PAGE_SIZE);
		uint32_t pbLen = FLASH_PAGE_SIZE - addrInPage;
		if(pbLen > len)
			pbLen = len;
		memcpy(pBuf + addrInPage, buf, pbLen);
		len -= pbLen;
		buf += pbLen;

		bool res = writePage(pageAddr, pBuf);
		free(pBuf);
		if(!res)
			return false;
		
		addr = pageAddr + FLASH_PAGE_SIZE;
	}
	
	// now write whole pages
	while(len >= FLASH_PAGE_SIZE)
	{
		if(!writePage(addr, buf))
			return false;
		buf += FLASH_PAGE_SIZE;
		len -= FLASH_PAGE_SIZE;
		addr += FLASH_PAGE_SIZE;
	}
	
	// and finally write last data part, if any
	if(len)
	{
		uint8_t *pBuf = (uint8_t *)malloc(FLASH_PAGE_SIZE);
		memcpy(pBuf, (void *)(EEPROM_START + addr), FLASH_PAGE_SIZE);
		memcpy(pBuf, buf, len);
		bool res = writePage(addr, pBuf);
		free(pBuf);
		if(!res)
			return false;
	}
	
	return true;
}

// read data from eeprom
uint32_t FishinoEEPROMClass::read32u(uint32_t addr)
{
	return *(uint32_t *)(EEPROM_START + (addr & 0xfffffffc));
}

uint16_t FishinoEEPROMClass::read16u(uint32_t addr)
{
	return *(uint16_t *)(EEPROM_START + (addr & 0xfffffffe));
}

uint8_t FishinoEEPROMClass::read8u(uint32_t addr)
{
	return *(uint8_t *)(EEPROM_START + addr);
}

bool FishinoEEPROMClass::read(uint32_t addr, uint8_t *buf, size_t len)
{
	memcpy(buf, (uint8_t *)(EEPROM_START + addr), len);
	return true;
}

FishinoEEPROMClass FishinoEEPROM;
