//////////////////////////////////////////////////////////////////////////////////////
//																					//
//								FishinoEEPROM.h										//
//				EEPROM routines for Fishino 8 bit and 32 bit boards					//
//					Created by Massimo Del Fedele, 2019								//
//																					//
//  Copyright (c) 2019 Massimo Del Fedele.  All rights reserved.					//
//																					//
//	Redistribution and use in source and binary forms, with or without				//
//	modification, are permitted provided that the following conditions are met:		//
//																					//
//	- Redistributions of source code must retain the above copyright notice,		//
//	  this list of conditions and the following disclaimer.							//
//	- Redistributions in binary form must reproduce the above copyright notice,		//
//	  this list of conditions and the following disclaimer in the documentation		//
//	  and/or other materials provided with the distribution.						//
//																					//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"		//
//	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE		//
//	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE		//
//	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE		//
//	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR				//
//	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF			//
//	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS		//
//	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN			//
//	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)			//
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE		//
//	POSSIBILITY OF SUCH DAMAGE.														//
//																					//
//	VERSION 7.8.0 - INITIAL VERSION													//
//	Version 8.0.0 - 26/07/2020 - UPDATED FOR FIRMWARE 8.0.0							//
//																					//
//////////////////////////////////////////////////////////////////////////////////////
#ifndef __FISHINOEEPROM_FISHINOEEPROM_H
#define __FISHINOEEPROM_FISHINOEEPROM_H

#include <Arduino.h>

class FishinoEEPROMClass
{
	private:
		
#ifdef _FISHINO_PIC32_
		// write a full eeprom page
		static bool writePage(uint32_t addr, void const *buf);

		// erase an eeprom page
		static bool erasePage(uint32_t addr);
#endif
		
	protected:

	public:
		
#ifdef _FISHINO_PIC32_
		// get page size in bytes
		static size_t pageSize(void);
		
		// get row size in bytes
		static size_t rowSize(void);
#endif

		// get total EEPROM size, in bytes
		static size_t size(void);
		
		// erase all EEPROM area
		static bool eraseAll(void);
		
		// erase an EEPROM area
		static bool eraseArea(uint32_t addr, uint32_t len);
		
		// write data to eeprom
		static bool write32u(uint32_t addr, uint32_t data);
		static bool write16u(uint32_t addr, uint16_t data);
		static bool write8u(uint32_t addr, uint8_t data);
		static bool write32s(uint32_t addr, int32_t data)
			{ return write32u(addr, (uint32_t)data); }
		static bool write16s(uint32_t addr, int16_t data)
			{ return write16u(addr, (uint16_t)data); }
		static bool write8s(uint32_t addr, int8_t data)
			{ return write8u(addr, (uint8_t)data); }
		static bool write(uint32_t addr, uint8_t const *buf, size_t len);

		static bool writeDouble(uint32_t addr, double d)
			{ return write(addr, (uint8_t const *)&d, sizeof(double)); }
		
		// read data from eeprom
		static uint32_t read32u(uint32_t addr);
		static uint16_t read16u(uint32_t addr);
		static uint8_t read8u(uint32_t addr);
		static int32_t read32s(uint32_t addr)
			{ return (int32_t)read32u(addr); }
		static int16_t read16s(uint32_t addr)
			{ return (int16_t)read16u(addr); }
		static int8_t read8s(uint32_t addr)
			{ return (int8_t)read8u(addr); }
		static bool read(uint32_t addr, uint8_t *buf, size_t len);
		
		static double readDouble(uint32_t addr)
			{ double d; if(read(addr, (uint8_t *)&d, sizeof(double))) return d; else return 0; }
};

extern FishinoEEPROMClass FishinoEEPROM;

#endif
