//////////////////////////////////////////////////////////////////////////////////////
//																					//
//							FishinoEEPROM_Atmega.cpp								//
//				EEPROM routines for Fishino 8 bit and 32 bit boards					//
//					Created by Massimo Del Fedele, 2019								//
//																					//
//  Copyright (c) 2019 Massimo Del Fedele.  All rights reserved.					//
//																					//
//	Redistribution and use in source and binary forms, with or without				//
//	modification, are permitted provided that the following conditions are met:		//
//																					//
//	- Redistributions of source code must retain the above copyright notice,		//
//	  this list of conditions and the following disclaimer.							//
//	- Redistributions in binary form must reproduce the above copyright notice,		//
//	  this list of conditions and the following disclaimer in the documentation		//
//	  and/or other materials provided with the distribution.						//
//																					//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"		//
//	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE		//
//	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE		//
//	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE		//
//	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR				//
//	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF			//
//	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS		//
//	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN			//
//	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)			//
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE		//
//	POSSIBILITY OF SUCH DAMAGE.														//
//																					//
//	VERSION 7.8.0 - INITIAL VERSION													//
//	Version 8.0.0 - 26/07/2020 - UPDATED FOR FIRMWARE 8.0.0							//
//																					//
//////////////////////////////////////////////////////////////////////////////////////
#include "FishinoEEPROM.h"

#ifdef __AVR__

#include <inttypes.h>
#include <avr/eeprom.h>

// get total EEPROM size, in bytes
size_t FishinoEEPROMClass::size(void)
{
#if defined(_FISHINO_UNO_) || defined(_FISHINO_UNO_R2_) || defined(_FISHINO_GUPPY_)
	return 1024;
#elif defined(_FISHINO_MEGA_) || defined(_FISHINO_MEGA_R2_)
	return 4096;
#else
	#error "Unsupported board"
#endif
}

// erase all EEPROM area
bool FishinoEEPROMClass::eraseAll(void)
{
	// we fake erase filling eeprom with 0xff values
	// as in flash-eeprom behaviour
	uint8_t buf[64];
	memset(buf, 0xff, 64);
	for(size_t i = 0; i < size() / 64; i++)
		eeprom_write_block(buf, (void *)(i * 64), 64);
	return true;
}

// erase an EEPROM area
bool FishinoEEPROMClass::eraseArea(uint32_t addr, uint32_t len)
{
	// we fake erase filling eeprom with 0xff values
	// as in flash-eeprom behaviour
	uint8_t buf[64];
	memset(buf, 0xff, 64);
	while(len > 64)
	{
		eeprom_write_block(buf, (void *)addr, 64);
		addr += 64;
		len -= 64;
	}
	if(len)
		eeprom_write_block(buf, (void *)addr, len);
	return true;
}

// write data to eeprom
bool FishinoEEPROMClass::write32u(uint32_t addr, uint32_t data)
{
	if(!(addr & 0x03))
		eeprom_write_dword((uint32_t *)addr, data);
	else
		eeprom_write_block(&data, (void *)addr, sizeof(uint32_t));
	return true;
}

bool FishinoEEPROMClass::write16u(uint32_t addr, uint16_t data)
{
	if(!(addr & 0x01))
		eeprom_write_word((uint16_t *)addr, data);
	else
		eeprom_write_block(&data, (void *)addr, sizeof(uint16_t));
	return true;
}

bool FishinoEEPROMClass::write8u(uint32_t addr, uint8_t data)
{
	eeprom_write_byte((uint8_t *)addr, data);
	return true;
}

bool FishinoEEPROMClass::write(uint32_t addr, uint8_t const *buf, size_t len)
{
	eeprom_write_block(buf, (void *)addr, len);
	return true;
}


// read data from eeprom
uint32_t FishinoEEPROMClass::read32u(uint32_t addr)
{
	if(!(addr & 0x03))
		return eeprom_read_dword((uint32_t const *)addr);

	uint32_t res;
	eeprom_read_block(&res, (const void *)addr, sizeof(uint32_t));
	return res;
}

uint16_t FishinoEEPROMClass::read16u(uint32_t addr)
{
	if(!(addr & 0x01))
		return eeprom_read_word((uint16_t const *)addr);

	uint16_t res;
	eeprom_read_block(&res, (const void *)addr, sizeof(uint16_t));
	return res;
}

uint8_t FishinoEEPROMClass::read8u(uint32_t addr)
{
	return eeprom_read_byte((uint8_t const *)addr);
}

bool FishinoEEPROMClass::read(uint32_t addr, uint8_t *buf, size_t len)
{
	eeprom_read_block(buf, (const void *)addr, len);
	return true;
}

FishinoEEPROMClass FishinoEEPROM;

#endif
