/////////////////////////////////////////////////////////////////
// Repeating Wifi Web Client                                   //
//                                                             //
// This sketch connects to a a web server and makes a request  //
// using an Arduino Wifi shield.                               //
//                                                             //
// Circuit:                                                    //
//   None                                                      //
//                                                             //
// created 23 April 2012                                       //
// modified 31 May 2012 by Tom Igoe                            //
// modified 13 Jan 2014 by Federico Vanzati                    //
// http://arduino.cc/en/Tutorial/WifiWebClientRepeating        //
// This code is in the public domain.                          //
//                                                             //
// adapted to Fishino 16 Ago 2015 by Massimo Del Fedele        //
/////////////////////////////////////////////////////////////////
#include<FishinoFtpClient.h>

#define DEBUG_LEVEL_INFO
#include <FishinoDebug.h>

FishinoFtpClient ftpClient;

SdFat sd;

//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////
// CONFIGURATION DATA		-- ADAPT TO YOUR NETWORK !!!
// DATI DI CONFIGURAZIONE	-- ADATTARE ALLA PROPRIA RETE WiFi !!!
#ifndef __MY_NETWORK_H

// here pur SSID of your network
// inserire qui lo SSID della rete WiFi
#define MY_SSID	""

// here put PASSWORD of your network. Use "" if none
// inserire qui la PASSWORD della rete WiFi -- Usare "" se la rete non ￨ protetta
#define MY_PASS	""

// here put required IP address (and maybe gateway and netmask!) of your Fishino
// comment out this lines if you want AUTO IP (dhcp)
// NOTE : if you use auto IP you must find it somehow !
// inserire qui l'IP desiderato ed eventualmente gateway e netmask per il fishino
// commentare le linee sotto se si vuole l'IP automatico
// nota : se si utilizza l'IP automatico, occorre un metodo per trovarlo !
#define IPADDR	192, 168,   1, 251
#define GATEWAY	192, 168,   1, 1
#define NETMASK	255, 255, 255, 0

#endif
//                    END OF CONFIGURATION DATA                      //
//                       FINE CONFIGURAZIONE                         //
///////////////////////////////////////////////////////////////////////

// define ip address if required
// NOTE : if your network is not of type 255.255.255.0 or your gateway is not xx.xx.xx.1
// you should set also both netmask and gateway
#ifdef IPADDR
	IPAddress ip(IPADDR);
	#ifdef GATEWAY
		IPAddress gw(GATEWAY);
	#else
		IPAddress gw(ip[0], ip[1], ip[2], 1);
	#endif
	#ifdef NETMASK
		IPAddress nm(NETMASK);
	#else
		IPAddress nm(255, 255, 255, 0);
	#endif
#endif

// the ftp server
const char *ftpServer = "ftp.drivehq.com";
const char *ftpUser = "cooter70";
const char *ftpPass = "ratiotec";

// test file name
const char *testFileName = "TestFile.txt";

void printWifiStatus()
{
	// print the SSID of the network you're attached to:
	// stampa lo SSID della rete:
	Serial.print("SSID: ");
	Serial.println(Fishino.SSID());

	// print your WiFi shield's IP address:
	// stampa l'indirizzo IP della rete:
	IPAddress ip = Fishino.localIP();
	Serial << F("IP Address: ");
	Serial.println(ip);

	// print the received signal strength:
	// stampa la potenza del segnale di rete:
	long rssi = Fishino.RSSI();
	Serial << F("signal strength (RSSI):");
	Serial.print(rssi);
	Serial << F(" dBm\n");
}

// create a test file on sd card
bool createTestFile(void)
{
	char buf[50];
	
	SdFile file;
	if(!file.open(testFileName, O_CREAT | O_WRITE | O_TRUNC))
	{
		DEBUG_ERROR("Failed to create file '%s'\n", testFileName);
		return false;
	}
	unsigned i = 0;
	for(int row = 0; row < 50; row++)
	{
		for(int col = 0; col < 10; col++)
		{
			sprintf(buf, "%08u ", i++);
			file.print(buf);
		}
		file.println();
	}
	file.close();
	return true;	
}

// list remote files on current remote directory
void listRemoteFiles(FishinoFtpClient &ftpClient)
{
	std::vector<FtpFileInfo> files;
	DEBUG_INFO("Listing files on current FTP server's directory\n");
	if(!ftpClient.list(files))
	{
		DEBUG_ERROR("Error listing current FTP server's directory\n");
		return;
	}
	
	// print all files
	DEBUG_INFO("Files on current FTP server's directory:\n");
	for(size_t i = 0; i < files.size(); i++)
		DEBUG_INFO_N("%03u:(%4s) %-20s (%8u bytes) %02d/%02d/%04d %02d:%02d:%02d\n",
			i, 
			files[i].dir ? "dir" : "file",
			files[i].name.c_str(),
			files[i].size,
			files[i].dateTime.day(),
			files[i].dateTime.month(),
			files[i].dateTime.year(),
			files[i].dateTime.hour(),
			files[i].dateTime.minute(),
			files[i].dateTime.second()
		);
}

void setup()
{
	// Initialize serial and wait for port to open
	// Inizializza la porta seriale e ne attende l'apertura
	Serial.begin(115200);
	
	// only for Leonardo needed
	// necessario solo per la Leonardo
	while (!Serial)
		;

	// reset and test WiFi module
	// resetta e testa il modulo WiFi
	while(!Fishino.reset())
		Serial << F("Fishino RESET FAILED, RETRYING...\n");
	Serial << F("Fishino WiFi RESET OK\n");
	
	Fishino.setPhyMode(PHY_MODE_11N);

	// go into station mode
	// imposta la modalità stazione
	Fishino.setMode(STATION_MODE);

	// try forever to connect to AP
	// tenta la connessione finchè non riesce
	Serial << F("Connecting to AP...");
	while(!Fishino.begin(MY_SSID, MY_PASS))
	{
		Serial << ".";
		delay(2000);
	}
	Serial << "OK\n";
	
	
	// setup IP or start DHCP client
	// imposta l'IP statico oppure avvia il client DHCP
#ifdef IPADDR
	Fishino.config(ip, gw, nm);
#else
	Fishino.staStartDHCP();
#endif

	// wait till connection is established
	Serial << F("Waiting for IP...");
	while(Fishino.status() != STATION_GOT_IP)
	{
		Serial << ".";
		delay(500);
	}
	Serial << "OK\n";

	// print connection status on serial port
	// stampa lo stato della connessione sulla porta seriale
	printWifiStatus();
	
	// sync RTC using ntp server
	// this is needed to have file date/time to work
	DEBUG_INFO_N("\n");
	DEBUG_INFO("Adjusting RTC time\n");
	DEBUG_INFO("(this may take some time and also fail!)\n");
	
	// epoch query may fail, so we try for a while
	uint32_t epoch;
	for(int i = 0; i < 10; i++)
	{
		epoch = Fishino.ntpEpoch();
		if(epoch)
			break;
		delay(1000);
	}
	if(!epoch)
		DEBUG_ERROR("Failed to get time from NTP server\n");
	else
	{
		DEBUG_INFO("Epoch : %u\n", epoch);
		DateTime now(epoch, DateTime::EPOCH_NTP);
		RTC.adjust(now);
		DEBUG_INFO("Current time : %02d/%02d/%04d - %02d:%02d\n",
			now.day(),
			now.month(),
			now.year(),
			now.hour(),
			now.minute()
		);
	}

	// initialize SD card
	DEBUG_INFO_N("\n");
	DEBUG_INFO("Initialize SD card\n");
	if (!sd.begin(SDCS, SD_SCK_MHZ(8)))
	{
		DEBUG_ERROR("Error initializing SD card\n");
		return;
	}
	
	// create a test file on sd card
	// just to have something to upload to server
	DEBUG_INFO_N("\n");
	DEBUG_INFO("Creating a test file on sd card\n");
	if(!createTestFile())
	{
		DEBUG_ERROR("Failed to create test file\n");
		return;
	}

	// connect to ftp server
	DEBUG_INFO_N("\n");
	DEBUG_INFO("Connecting to FTP server\n");
	if(!ftpClient.connect(ftpServer, ftpUser, ftpPass))
	{
		DEBUG_ERROR("Connection failed\n");
		return;
	}
	
	// create a test directory
	DEBUG_INFO_N("\n");
	DEBUG_INFO("Creating 'TestDir' remote directory\n");
	if(!ftpClient.mkDir("/TestDir"))
		DEBUG_ERROR("Failed creating 'TestDir' remote directory\n");
	else
		DEBUG_INFO("Successfully created 'TestDir' remote directory\n");

	// go to TestDir directory
	DEBUG_INFO_N("\n");
	DEBUG_INFO("Entering 'TestDir' remote directory\n");
	if(!ftpClient.setCurDir("/TestDir"))
		DEBUG_ERROR("Failed entering 'TestDir' remote directory\n");
	else
		DEBUG_INFO("Successfully entered 'TestDir' remote directory\n");

	// get current directory
	DEBUG_INFO_N("\n");
	DEBUG_INFO("Getting current remote directory\n");
	String curDir = ftpClient.getCurDir();
	DEBUG_INFO("Current remote directory is : '%s'\n", curDir.c_str());
	
	// upload something to remote directory
	DEBUG_INFO_N("\n");
	DEBUG_INFO("Uploading some files to remote directory\n");
	if(!ftpClient.upload(testFileName, "file1.txt"))
		DEBUG_ERROR("Failed to upload 'file1.txt'\n");
	else
		DEBUG_INFO("Succesfully uploaded 'file1.txt'\n");

	if(!ftpClient.upload(testFileName, "file2.txt"))
		DEBUG_ERROR("Failed to upload 'file2.txt'\n");
	else
		DEBUG_INFO("Succesfully uploaded 'file2.txt'\n");
	
	if(!ftpClient.upload(testFileName, "file3.txt"))
		DEBUG_ERROR("Failed to upload 'file3.txt'\n");
	else
		DEBUG_INFO("Succesfully uploaded 'file3.txt'\n");

	// list files '/TestDir' remote directory
	DEBUG_INFO_N("\n");
	DEBUG_INFO("List files on '/TestDir' remote directory\n");
	listRemoteFiles(ftpClient);
	
	// rename remote file3.txt to file4.txt
	DEBUG_INFO_N("\n");
	DEBUG_INFO("Renaming remote file 'file3.txt' to 'file4.txt'\n");
	if(!ftpClient.rename("file3.txt", "file4.txt"))
		DEBUG_ERROR("Failed to rename remote file 'file3.txt' to 'file4.txt'\n");
	else
		DEBUG_INFO("Successfully renamed remote file 'file3.txt' to 'file4.txt'\n");
	
	// list again the remote directory
	DEBUG_INFO_N("\n");
	DEBUG_INFO("List files on '/TestDir' remote directory after rename\n");
	listRemoteFiles(ftpClient);
	
	// re-download files from remote directory
	DEBUG_INFO_N("\n");
	DEBUG_INFO("Re-download the 3 files from remote server\n");
	if(!ftpClient.download("file1.txt", "/file1.txt"))
		DEBUG_ERROR("Failed to download 'file1.txt'\n");
	else
		DEBUG_INFO("Successfully downloaded 'file1.txt'\n");
	
	if(!ftpClient.download("file2.txt", "/file2.txt"))
		DEBUG_ERROR("Failed to download 'file2.txt'\n");
	else
		DEBUG_INFO("Successfully downloaded 'file2.txt'\n");
	
	if(!ftpClient.download("file4.txt", "/file4.txt"))
		DEBUG_ERROR("Failed to download 'file4.txt'\n");
	else
		DEBUG_INFO("Successfully downloaded 'file4.txt'\n");
	
	// check if files are there
	DEBUG_INFO_N("\n");
	DEBUG_INFO("Check if downloaded files are on SD card\n");
	SdFile f;
	if(!f.open("/file1.txt"))
		DEBUG_ERROR("Missing local file 'file1.txt'\n");
	else
	{
		DEBUG_INFO("Local file 'file1.txt' found\n");
		f.close();
		sd.remove("/file1.txt");
	}
	if(!f.open("/file2.txt"))
		DEBUG_ERROR("Missing local file 'file2.txt'\n");
	else
	{
		DEBUG_INFO("Local file 'file2.txt' found\n");
		f.close();
		sd.remove("/file2.txt");
	}
	
	if(!f.open("/file4.txt"))
		DEBUG_ERROR("Missing local file 'file4.txt'\n");
	else
	{
		DEBUG_INFO("Local file 'file4.txt' found\n");
		DEBUG_INFO("File content:\n");
		while(f.available())
			DEBUG_INFO_N("%c", f.read());
		DEBUG_INFO_N("\n\n");
		f.close();
		sd.remove("/file4.txt");
	}
	
	// remove the uploaded files from remote directory
	DEBUG_INFO_N("\n");
	DEBUG_INFO("Remove uploaded files from remote directory\n");
	if(!ftpClient.deleteFile("file1.txt"))
		DEBUG_ERROR("Failed to remove 'file1.txt' from remote directory\n");
	else
		DEBUG_INFO("Succesfully removed 'file1.txt' from remote directory\n");

	if(!ftpClient.deleteFile("file2.txt"))
		DEBUG_ERROR("Failed to remove 'file2.txt' from remote directory\n");
	else
		DEBUG_INFO("Succesfully removed 'file2.txt' from remote directory\n");

	if(!ftpClient.deleteFile("file4.txt"))
		DEBUG_ERROR("Failed to remove 'file4.txt' from remote directory\n");
	else
		DEBUG_INFO("Succesfully removed 'file4.txt' from remote directory\n");

	// list files '/TestDir' remote directory again
	DEBUG_INFO_N("\n");
	DEBUG_INFO("List files on '/TestDir' remote directory again\n");
	listRemoteFiles(ftpClient);
	
	// go back to root directory
	DEBUG_INFO_N("\n");
	DEBUG_INFO("Returning to root remote directory\n");
	if(!ftpClient.setCurDir("/"))
		DEBUG_ERROR("Failed returning to root remote directory\n");
	else
		DEBUG_INFO("Successfully returned to root remote directory\n");

	// list files root remote directory
	DEBUG_INFO_N("\n");
	DEBUG_INFO("List files on root remote directory\n");
	listRemoteFiles(ftpClient);
	
	// remove the 'TestDir' remote directory
	DEBUG_INFO_N("\n");
	DEBUG_INFO("Deleting 'TestDir' remote directory\n");
	if(!ftpClient.rmDir("/TestDir"))
		DEBUG_ERROR("Failed deleting 'TestDir' remote directory\n");
	else
		DEBUG_INFO("Successfully deleted 'TestDir' remote directory\n");
	
	// list files root remote directoryagain
	DEBUG_INFO_N("\n");
	DEBUG_INFO("List files on root remote directory\n");
	listRemoteFiles(ftpClient);

	// close the connection
	DEBUG_INFO_N("\n");
	DEBUG_INFO("Closing connection\n");
	ftpClient.disconnect();
}

void loop()
{
}
