//////////////////////////////////////////////////////////////////////////////////////
//																					//
//							FishinoFtpClient.h										//
//				FTP passive client library for Fishino boards						//
//					Created by Massimo Del Fedele, 2018								//
//																					//
//  Copyright (c) 2018 Massimo Del Fedele.  All rights reserved.					//
//																					//
//	Redistribution and use in source and binary forms, with or without				//
//	modification, are permitted provided that the following conditions are met:		//
//																					//
//	- Redistributions of source code must retain the above copyright notice,		//
//	  this list of conditions and the following disclaimer.							//
//	- Redistributions in binary form must reproduce the above copyright notice,		//
//	  this list of conditions and the following disclaimer in the documentation		//
//	  and/or other materials provided with the distribution.						//
//																					//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"		//
//	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE		//
//	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE		//
//	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE		//
//	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR				//
//	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF			//
//	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS		//
//	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN			//
//	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)			//
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE		//
//	POSSIBILITY OF SUCH DAMAGE.														//
//																					//
//	VERSION 1.0.0 - INITIAL VERSION													//
//	Version 8.0.0 - 26/07/2020 - UPDATED FOR FIRMWARE 8.0.0							//
//																					//
//////////////////////////////////////////////////////////////////////////////////////
#include "FishinoFtpClient.h"

// enable INFO level to display FTP protocol commands
// enable ERROR level to display only FTP errors
// comment both if you want to spare memory

//#define DEBUG_LEVEL_INFO
#define DEBUG_LEVEL_ERROR
#include <FishinoDebug.h>

#include "ftpparse.h"

// File system object.
extern SdFat sd;

// constructor
FishinoFtpClient::FishinoFtpClient()
{
	_connected = false;
	_server = NULL;
	
	_respCode = 0;
	_response[0] = 0;
	_respCount = 0;
	
	_errCode = FishinoFtpErrors::OK;
	
	// timeout defaults to 2 seconds
	_timeout = 2000;
}

// destructor
FishinoFtpClient::~FishinoFtpClient()
{
	if(_connected)
		disconnect();
}

// set global timeout
void FishinoFtpClient::setTimeout(uint32_t tim)
{
	_timeout = tim;
}

// flush cmd client after a command
void FishinoFtpClient::flushCmdClient(bool longWait)
{
	DEBUG_INFO("Flushing cmd client\n");
	
	uint32_t tim;
	if(longWait)
		tim = millis() + _timeout;
	else
		tim = millis() + 50;
	while (!_cmdClient.available())
	{
		if(millis() > tim)
		{
			if(longWait)
				DEBUG_ERROR("Timeout flushing ftp command client\n");
			else
				DEBUG_INFO("Timeout flushing ftp command client\n");
			return;
		}
		delay(1);
	}

	DEBUG_INFO("Client response : ");
	while (_cmdClient.available())
	{
#ifdef DEBUG_LEVEL_INFO
		uint8_t thisByte = _cmdClient.read();
		DEBUG_INFO_N("%c", thisByte);
#else
		_cmdClient.read();
#endif
	}
	DEBUG_INFO_N("\n");
}

// abort ftp connection on errors
void FishinoFtpClient::abort(void)
{
	// do nothing if not connected
	if(!_connected || !_cmdClient.connected())
		return;

	// send a quit command to server
	_cmdClient.println(F("QUIT"));
	
	// reset connection flag
	_connected = false;
	_server = NULL;
	
	// flush any client's response
	flushCmdClient(false);

	_cmdClient.stop();
	DEBUG_INFO("Disconnected from FTP server\n");
}

// read command response from ftp server
bool FishinoFtpClient::respRead(void)
{
	// clear previous response
	_respCode = 0;
	_response[0] = 0;
	_respCount = 0;
	
	uint8_t thisByte;

	// wait for server's response
	uint32_t tim = millis() + _timeout;
	while (!_cmdClient.available())
	{
		if(millis() > tim)
		{
			DEBUG_ERROR("Timeout waiting for response from server\n");
			_errCode = FishinoFtpErrors::TIMEOUT;
			return false;
		}
		delay(1);
	}

	// read response code
	_respCode = _cmdClient.read();
	DEBUG_INFO("Response code : %d\n", _respCode);

	// read rest of response data
	DEBUG_INFO("Response data : ");
	while (_cmdClient.available())
	{
		thisByte = _cmdClient.read();
		DEBUG_INFO_N("%c", thisByte);
		if(_respCount < 127)
		{
			_response[_respCount++] = thisByte;
			_response[_respCount] = 0;
		}
	}
	DEBUG_INFO_N("\n");

	// check error code
	if (_respCode >= '4')
	{
//		abort();
		return false;
	}

	return true;
}

// start a passive connection
// (opening the data stream)
bool FishinoFtpClient::pasv(void)
{
	// enter passive mode
	DEBUG_INFO("PASV\n");
	_cmdClient.println(F("PASV"));
	if(!respRead())
	{
		_errCode = FishinoFtpErrors::PASV_COMMAND_FAILED;
		DEBUG_ERROR("PASV COMMAND FAILED\n");
		return false;
	}

	// parse ftp PASV response (a1, a2, a3, a4, p1, p2)
	uint8_t pasv[6];
	char *tStr = strtok((char *)_response, "(,");
	for (int i = 0; i < 6; i++)
	{
		tStr = strtok(NULL, "(,");
		pasv[i] = atoi(tStr);
		if (tStr == NULL)
		{
			DEBUG_ERROR("Bad PASV Answer : '%s'\n", _response);
			Serial.println(F("Bad PASV Answer"));
			_errCode = FishinoFtpErrors::BAD_PASV_ANSWER;
			_cmdClient.stop();
			return false;
		}
	}

	// get data port from PASV answer
	uint16_t dataPort = (((uint16_t)pasv[4]) << 8) | pasv[5];
	DEBUG_INFO("Data port : %u\n", dataPort);

	// try to connect on data port
	if(!_dataClient.connect(_server, dataPort))
	{
		DEBUG_ERROR("Failed to establish data connection\n");
		_cmdClient.stop();
		_errCode = FishinoFtpErrors::DATA_CONNECTION_FAILED;
		return false;
	}
	DEBUG_INFO("Data connection on port %u succesfully established\n", dataPort);
	return true;
}

// connect to server
bool FishinoFtpClient::connect(const char *srv, const char *user, const char *pwd)
{
	// reset error code
	_errCode = FishinoFtpErrors::OK;
	
	// if already connected, disconnect first
	if(_connected)
		disconnect();
	
	// open command channel
	if(!_cmdClient.connect(srv, 21))
	{
		DEBUG_ERROR("Failed to connect to server '%s'\n", srv);
		_errCode = FishinoFtpErrors::CONNECTION_ERROR;
		return false;
	}
	DEBUG_INFO("Command connection established\n");
	
	// send credentials to FTP server
	DEBUG_INFO("USER %s\n", user);
	_cmdClient.print(F("USER "));
	_cmdClient.println(user);
	if(!respRead())
	{
		_errCode = FishinoFtpErrors::USER_COMMAND_FAILED;
		DEBUG_ERROR("USER COMMAND FAILED\n");
		return false;
	}
	DEBUG_INFO("PASS %s\n", pwd);
	_cmdClient.print(F("PASS "));
	_cmdClient.println(pwd);
	if(!respRead())
	{
		_errCode = FishinoFtpErrors::PASS_COMMAND_FAILED;
		DEBUG_ERROR("PASS COMMAND FAILED\n");
		return false;
	}

	// query system type
	// (don't know if that's needed...)
	DEBUG_INFO("SYST\n");
	_cmdClient.println(F("SYST"));
	if(!respRead())
	{
		_errCode = FishinoFtpErrors::SYST_COMMAND_FAILED;
		DEBUG_ERROR("SYST COMMAND FAILED\n");
		return false;
	}

	// set data type to binary
	DEBUG_INFO("TYPE I\n");
	_cmdClient.println(F("TYPE I"));
	if(!respRead())
	{
		_errCode = FishinoFtpErrors::TYPE_COMMAND_FAILED;
		DEBUG_ERROR("TYPE COMMAND FAILED\n");
		return false;
	}

	// signal connection ok
	_connected = true;
	_server = srv;
	return true;
}

// disconnet from server
bool FishinoFtpClient::disconnect(void)
{
	// if not connected, just do nothing
	if(!_connected)
		return true;

	_connected = false;
	_server = NULL;
	
	// close data client
	_dataClient.stop();
	DEBUG_INFO("Data client disconnected\n");
	
	// flush any command answer before closing cmd client
	flushCmdClient(false);

	// send a QUIT command to server
	DEBUG_INFO("QUIT\n");
	_cmdClient.println(F("QUIT"));
	if (!respRead())
		return false;

	// close command client and leave
	_cmdClient.stop();
	DEBUG_INFO("Command client disconnected\n");
	
	return true;
}

// get current directory on server
String FishinoFtpClient::getCurDir(void)
{
	String res;
	
	// send a list command
	DEBUG_INFO("PWD\n");
	_cmdClient.println(F("PWD"));
	if(!respRead())
	{
		DEBUG_ERROR("PWD command failed\n");
		_errCode = FishinoFtpErrors::PWD_FAILED;
		_dataClient.stop();
		return res;
	}
	
	// parse the response data to extract the directory
	const char *start = strchr((const char *)_response, '"');
	if(!start)
		return res;
	const char *end = strrchr((const char *)_response, '"');
	if(!end)
		return res;
	if(end - start <= 1)
		return res;
	char *path = strndup(start + 1, end - start - 1);
	res = path;
	free(path);

	return res;
}

// set current directory on server
bool FishinoFtpClient::setCurDir(const char *dir)
{
	DEBUG_INFO("CWD %s\n", dir);
	_cmdClient.print(F("CWD "));
	_cmdClient.println(dir);
	if(!respRead())
	{
		DEBUG_ERROR("CWD command failed\n");
		_errCode = FishinoFtpErrors::CWD_FAILED;
		_dataClient.stop();
		return false;
	}

	return true;
}

// create a directory on server
bool FishinoFtpClient::mkDir(const char *dir)
{
	DEBUG_INFO("MKD %s\n", dir);
	_cmdClient.print(F("MKD "));
	_cmdClient.println(dir);
	if(!respRead())
	{
		DEBUG_ERROR("MKD command failed\n");
		_errCode = FishinoFtpErrors::MKD_FAILED;
		_dataClient.stop();
		return false;
	}

	return true;
}

// remove an empty directory on server
bool FishinoFtpClient::rmDir(const char *dir)
{
	DEBUG_INFO("RMD %s\n", dir);
	_cmdClient.print(F("RMD "));
	_cmdClient.println(dir);
	if(!respRead())
	{
		DEBUG_ERROR("RMD command failed\n");
		_errCode = FishinoFtpErrors::RMD_FAILED;
		_dataClient.stop();
		return false;
	}

	return true;
}

// list files on current server's directory
bool FishinoFtpClient::list(std::vector<FtpFileInfo> &files)
{
	files.clear();
	
	// enter passive mode opening data connection
	if(!pasv())
		return false;

	String curLine;

	// send a list command
	DEBUG_INFO("LIST\n");
	_cmdClient.println(F("LIST"));
	if(!respRead())
	{
		DEBUG_ERROR("LIST command failed\n");
		_errCode = FishinoFtpErrors::LIST_FAILED;
		_dataClient.stop();
		return false;
	}
	
	uint32_t tim = millis() + _timeout;
	while (_dataClient.connected())
	{
		if(millis() > tim)
		{
			_dataClient.stop();
			_errCode = FishinoFtpErrors::TIMEOUT;
			DEBUG_ERROR("Timeout waiting for directory listing\n");
			return false;
		}
		while (_dataClient.available())
		{
			char c = _dataClient.read();
			bool done = false;
			if(c == '\r')
			{
				if(_dataClient.peek() == '\n')
					_dataClient.read();
				done = true;
			}
			else if(c == '\n')
			{
				if(_dataClient.peek() == '\r')
					_dataClient.read();
				done = true;
			}

			if(done)
			{
				struct ftpparse fp;
				
				// copy current line, as c_str() is a const
				// and line is modified by parser
				char *line = strdup(curLine.c_str());
				
				if(ftpparse(&fp, line, strlen(line)))
				{
					FtpFileInfo info;
					
					char *name = strndup(fp.name, fp.namelen);
					info.name = name;
					free(name);
					info.dir = fp.flagtrycwd;
					info.size = fp.size;
					info.dateTime = DateTime(fp.mtime, DateTime::EPOCH_UNIX);
					
					files.push_back(info);
					
				}
				free(line);
				curLine = "";
			}
			else
				curLine += c;

			tim = millis() + _timeout;
		}
	}
	_dataClient.stop();

	// flush any client's response
	flushCmdClient(true);

	return true;
	
}

// fetch a file from server
// 'file' must be opened in WRITE mode
bool FishinoFtpClient::download(const char *name, SdFile &file)
{
	// enter passive mode opening data connection
	if(!pasv())
		return false;

	DEBUG_INFO("RETR %s\n", name);
	_cmdClient.print(F("RETR "));
	_cmdClient.println(name);
	if(!respRead())
	{
		DEBUG_ERROR("RETR command failed\n");
		_errCode = FishinoFtpErrors::DOWNLOAD_FAILED;
		_dataClient.stop();
		return false;
	}

	DEBUG_INFO("Retrieving data from remote server\n");
	uint8_t buf[64];
	uint32_t tim = millis() + _timeout;
	while (_dataClient.connected())
	{
		if(millis() > tim)
		{
			_dataClient.stop();
			return false;
		}
		while (_dataClient.available())
		{
			int len = _dataClient.read(buf, 64);
			if(len)
				file.write(buf, len);
			tim = millis() + _timeout;
		}
	}

	DEBUG_INFO("Done retrieving data from remote server\n");

	// flush any client's response
	flushCmdClient(true);

	return true;
}

bool FishinoFtpClient::download(const char *name, const char *localPath)
{
	SdFile file;
	if(!file.open(localPath, O_CREAT | O_WRITE | O_TRUNC))
	{
		DEBUG_ERROR("Failed to create file '%s'\n", localPath);
		return false;
	}
	bool res = download(name, file);
	file.close();
	if(!res)
	{
		DEBUG_ERROR("Download failed - removing file '%s'\n", localPath);
		sd.remove(localPath);
		return false;
	}
	return true;
}
// upload a file to server
bool FishinoFtpClient::upload(SdFile &file, const char *remoteName)
{
	// enter passive mode opening data connection
	if(!pasv())
		return false;

	DEBUG_INFO("STOR %s\n", remoteName);
	_cmdClient.print(F("STOR "));
	_cmdClient.println(remoteName);
	if(!respRead())
	{
		DEBUG_ERROR("STOR command failed\n");
		_errCode = FishinoFtpErrors::UPLOAD_FAILED;
		_dataClient.stop();
		return false;
	}

	DEBUG_INFO("Sending data to remote host\n");
	uint8_t buf[64];
	while(file.available())
	{
		uint8_t len = file.read(buf, 64);
		if(len)
		{
			uint32_t written = _dataClient.write(buf, len);
			if( written != len)
			{
				DEBUG_ERROR("Error writing data to server, tried %u, sent %u\n", (unsigned)len, (unsigned)written);
				return false;
			}
		}
	}
	DEBUG_INFO("Done sending data to remote host\n");

	// close data client
	_dataClient.stop();

	// flush any client's response
	flushCmdClient(false);

	return true;
}

bool FishinoFtpClient::upload(const char *localPath, const char *remoteName)
{
	SdFile file;
	if(!file.open(localPath, O_READ))
	{
		DEBUG_ERROR("Failed to open local file '%s'\n", localPath);
		return false;
	}
	bool res = upload(file, remoteName);
	file.close();
	return res;
}

// delete a file from server
bool FishinoFtpClient::deleteFile(const char *name)
{
	DEBUG_INFO("DELE %s\n", name);
	_cmdClient.print(F("DELE "));
	_cmdClient.println(name);
	if(!respRead())
	{
		DEBUG_ERROR("DELE command failed\n");
		_errCode = FishinoFtpErrors::DELE_FAILED;
		_dataClient.stop();
		return false;
	}

	return true;
}

// rename a file/folder
bool FishinoFtpClient::rename(const char *oldName, const char *newName)
{
	DEBUG_INFO("RNFR %s\n", oldName);
	_cmdClient.print(F("RNFR "));
	_cmdClient.println(oldName);
	if(!respRead())
	{
		DEBUG_ERROR("RNFR command failed\n");
		_errCode = FishinoFtpErrors::RNFR_FAILED;
		_dataClient.stop();
		return false;
	}

	DEBUG_INFO("RNTO %s\n", newName);
	_cmdClient.print(F("RNTO "));
	_cmdClient.println(newName);
	if(!respRead())
	{
		DEBUG_ERROR("RNTO command failed\n");
		_errCode = FishinoFtpErrors::RNTO_FAILED;
		_dataClient.stop();
		return false;
	}

	return true;
}
