//////////////////////////////////////////////////////////////////////////////////////
//																					//
//							FishinoFtpClient.h										//
//				FTP passive client library for Fishino boards						//
//					Created by Massimo Del Fedele, 2018								//
//																					//
//  Copyright (c) 2018 Massimo Del Fedele.  All rights reserved.					//
//																					//
//	Redistribution and use in source and binary forms, with or without				//
//	modification, are permitted provided that the following conditions are met:		//
//																					//
//	- Redistributions of source code must retain the above copyright notice,		//
//	  this list of conditions and the following disclaimer.							//
//	- Redistributions in binary form must reproduce the above copyright notice,		//
//	  this list of conditions and the following disclaimer in the documentation		//
//	  and/or other materials provided with the distribution.						//
//																					//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"		//
//	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE		//
//	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE		//
//	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE		//
//	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR				//
//	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF			//
//	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS		//
//	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN			//
//	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)			//
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE		//
//	POSSIBILITY OF SUCH DAMAGE.														//
//																					//
//	VERSION 1.0.0 - INITIAL VERSION													//
//	Version 8.0.0 - 26/07/2020 - UPDATED FOR FIRMWARE 8.0.0							//
//																					//
//////////////////////////////////////////////////////////////////////////////////////
#ifndef __FISHINOFTPCLIENT_FISHINOFTPCLIENT_H
#define __FISHINOFTPCLIENT_FISHINOFTPCLIENT_H

#include <Fishino.h>
#include <FishinoSdFat.h>
#include <FishinoStl.h>

// for DateTime
#include <FishinoRTC.h>

// ftp error codes
enum class FishinoFtpErrors
{
	OK						= 0,
	SERVER_NOT_FOUND		= 1,
	CONNECTION_ERROR		= 2,
	
	USER_COMMAND_FAILED		= 3,
	PASS_COMMAND_FAILED		= 4,
	SYST_COMMAND_FAILED		= 5,
	TYPE_COMMAND_FAILED		= 6,
	PASV_COMMAND_FAILED		= 7,
	
	FILE_NOT_FOUND			= 20,
	BAD_PASV_ANSWER			= 21,
	DATA_CONNECTION_FAILED	= 22,
	DOWNLOAD_FAILED			= 23,
	UPLOAD_FAILED			= 24,
	LIST_FAILED				= 25,
	PWD_FAILED				= 26,
	CWD_FAILED				= 27,
	MKD_FAILED				= 28,
	RMD_FAILED				= 29,
	RNFR_FAILED				= 30,
	RNTO_FAILED				= 31,
	DELE_FAILED				= 32,
	
	TIMEOUT					= 99
};

struct FtpFileInfo
{
	bool dir:1;
	uint32_t size:31;
	String name;
	DateTime dateTime;
	
	FtpFileInfo() : dir(false), size(0), name(""), dateTime(0) {};
};

class FishinoFtpClient
{
	private:
		
		// the server (needed for pasv command)
		const char *_server;

		// FTP command and data clients
		FishinoClient _cmdClient, _dataClient;
		
		// last error code
		FishinoFtpErrors _errCode;
		
		// connected flag
		bool _connected;
		
		// response from server
		uint8_t _respCode;
		uint8_t _response[128];
		uint8_t _respCount;
		
		// flush cmd client after a command
		void flushCmdClient(bool longWait);
		
		// read command response from ftp server
		bool respRead(void);
		
		// abort ftp connection on errors
		void abort(void);
		
		// start a passive connection
		// (opening the data stream)
		bool pasv(void);
		
		// timeout value, in milliseconds
		uint32_t _timeout;

	protected:

	public:

		// constructor
		FishinoFtpClient();

		// destructor
		~FishinoFtpClient();
		
		// set/get global timeout
		void setTimeout(uint32_t tim);
		uint32_t getTimeout(void) const { return _timeout; }
		
		// get last error code
		FishinoFtpErrors getLastError(void) { return _errCode; }
		
		// connect to server
		bool connect(const char *srv, const char *user, const char *pwd);
		
		// disconnet from server
		bool disconnect(void);
		
		// get current directory on server
		String getCurDir(void);
		
		// set current directory on server
		bool setCurDir(const char *dir);
		
		// create a directory on server
		bool mkDir(const char *dir);
		
		// remove an empty directory on server
		bool rmDir(const char *dir);
		
		// list files on current server's directory
		bool list(std::vector<FtpFileInfo> &files);
		
		// fetch a file from server
		bool download(const char *name, SdFile &file);
		bool download(const char *name, const char *localPath);
		
		// upload a file to server
		bool upload(SdFile &file, const char *remoteName);
		bool upload(const char *localPath, const char *remoteName);
		
		// delete a file from server
		bool deleteFile(const char *name);
		
		// rename a file/folder
		bool rename(const char *oldName, const char *newName);
};

#endif
