/*
  FishinoLCD03.cpp - Arduino library for I2C LCD03 display from Robot Electronics
  see http://www.robot-electronics.co.uk/htm/Lcd03tech.htm
  Copyright (c) 2013 Ben Arblaster.  All right reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "FishinoLCD03.h"

#include <inttypes.h>
#include "Arduino.h"
#include <Wire.h>

char _i2c_address;

#ifdef ARDUINO_ARCH_esp826
uint8_t _sda_pin;
uint8_t _scl_pin;

FishinoLCD03::FishinoLCD03(char i2c_address, uint8_t sda_pin, uint8_t scl_pin) {
  _sda_pin = sda_pin;
  _scl_pin = scl_pin;
#else
FishinoLCD03::FishinoLCD03(char i2c_address) {
#endif
  // Convert 8-bit address from LCD to 7-bit address for Wire
  _i2c_address = i2c_address>>1;
}

static void BEGIN(void)
{
	Wire.begin();
	delay(10);
}

byte AVAILABLE(void)
{
	return Wire.available();
}

static void BEGINTRANSMISSION(byte addr)
{
	Wire.beginTransmission(addr);
	delayMicroseconds(30);
}

static void ENDTRANSMISSION(void)
{
	Wire.endTransmission();
}

static void REQUESTFROM(byte addr, byte num)
{
	Wire.requestFrom(addr, num);
}

static byte READ(void)
{
	return Wire.read();
}

static void WRITE(byte b)
{
	Wire.write(b);
	delayMicroseconds(100);
}


void FishinoLCD03::begin(uint8_t cols, uint8_t rows) {
#ifdef ARDUINO_ARCH_esp826
  BEGIN(_sda_pin, _scl_pin);
#else
  BEGIN();
#endif
  noCursor();
  clear();
}

void FishinoLCD03::setCursor(uint8_t pos) {
  pos++;
  BEGINTRANSMISSION(_i2c_address);
  WRITE(REG_COMMAND);
  WRITE(LCD_CURSORPOS);
  WRITE(pos);
  ENDTRANSMISSION();
}

void FishinoLCD03::setCursor(uint8_t col, uint8_t row) {
  // convert LiquidCrystal col & rows to LCD03
  col++;
  row++;
  BEGINTRANSMISSION(_i2c_address);
  WRITE(REG_COMMAND);
  WRITE(LCD_CURSORPOSXY);
  WRITE(row);
  WRITE(col);
  ENDTRANSMISSION();
}

void FishinoLCD03::tabSet(uint8_t ts) {
  BEGINTRANSMISSION(_i2c_address);
  WRITE(REG_COMMAND);
  WRITE(LCD_TABSET);
  WRITE(ts);
  ENDTRANSMISSION();
}

void FishinoLCD03::createChar(uint8_t location, uint8_t charmap[]) {
  // remap custom chars from 0-7 to match LiquidCrystal
  location += LCD_CUSTOMCHAR_BASE;
  BEGINTRANSMISSION(_i2c_address);
  WRITE(REG_COMMAND);
  WRITE(LCD_CUSTOMCHAR);
  WRITE(location);
  for(int i = 0; i < 8; i++) {
    WRITE(charmap[i] |= LCD_CUSTOMCHAR_MASK);
  }
  ENDTRANSMISSION();
}

// get the free buffer capacity in bytes
uint8_t FishinoLCD03::bufferFreeBytes() {
  REQUESTFROM(_i2c_address, 1);
  if(AVAILABLE()) {
    return READ();
  }
  return 0;
}

uint16_t FishinoLCD03::readKeypad () {  
  REQUESTFROM(_i2c_address, 3);
  uint8_t lowByte;
  uint8_t highByte;
  uint16_t bothBytes;
 
  // Don't care about the first byte
  READ();
  // Read the low and high bytes
  lowByte = READ();
  highByte = READ();

  bothBytes = (lowByte | highByte << 8);

  return bothBytes;
}

// write a single byte in one transmission
size_t FishinoLCD03::write(uint8_t value) {
  // remap custom chars to 0-7 to match LiquidCrystal
  if(value < 8) {
    value += LCD_CUSTOMCHAR_BASE;
  }
  send(value);
  // assume write was a success
  return 1;
}

// write multiple bytes in as many transmissions as required
size_t FishinoLCD03::write(const uint8_t *buffer, size_t size) {
  //wait until the buffer is empty enough
  while(bufferFreeBytes() < BUFFER_LENGTH)
  	;
  BEGINTRANSMISSION(_i2c_address);

  WRITE(REG_COMMAND);
  for(int i = 0; i < size; i++) {    
    // we've filled the I2C buffer, flush it and start a new transmission
    if(i != 0 && i % (BUFFER_LENGTH-1) == 0) {
      ENDTRANSMISSION();
      //wait until the buffer is empty enough
      while(bufferFreeBytes() < BUFFER_LENGTH);
      BEGINTRANSMISSION(_i2c_address);
      WRITE(REG_COMMAND);
    }
    WRITE(buffer[i]);
  }
  ENDTRANSMISSION();
  // assume write was a success
  return size;
}

// send a single byte command in a single transmission
void FishinoLCD03::send(uint8_t command) {
  BEGINTRANSMISSION(_i2c_address);
  WRITE(REG_COMMAND);
  WRITE(command);
  ENDTRANSMISSION();
}
