//////////////////////////////////////////////////////////////////////////////////////
//																					//
//								FishinoTimer32.cpp									//
//	Library to handle Fishino's timers in uniform way between 8 and 32 bit boards	//
//					Created by Massimo Del Fedele, 2019								//
//																					//
//  Copyright (c) 2019 Massimo Del Fedele.  All rights reserved.					//
//																					//
//	Redistribution and use in source and binary forms, with or without				//
//	modification, are permitted provided that the following conditions are met:		//
//																					//
//	- Redistributions of source code must retain the above copyright notice,		//
//	  this list of conditions and the following disclaimer.							//
//	- Redistributions in binary form must reproduce the above copyright notice,		//
//	  this list of conditions and the following disclaimer in the documentation		//
//	  and/or other materials provided with the distribution.						//
//																					//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"		//
//	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE		//
//	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE		//
//	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE		//
//	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR				//
//	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF			//
//	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS		//
//	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN			//
//	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)			//
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE		//
//	POSSIBILITY OF SUCH DAMAGE.														//
//																					//
//	VERSION 7.6.0 - INITIAL VERSION													//
//	Version 8.0.0 - 26/07/2020 - UPDATED FOR FIRMWARE 8.0.0							//
//																					//
//////////////////////////////////////////////////////////////////////////////////////
#include <FishinoTimer.h>

#ifdef _FISHINO_PIC32_

static FishinoTimerClass *_timer1_instance = NULL;
void __attribute__((interrupt(), nomips16)) _timer1InterruptHandler(void)
{
	_timer1_instance->isr();
}

static FishinoTimerClass *_timer2_instance = NULL;
void __attribute__((interrupt(), nomips16)) _timer2InterruptHandler(void)
{
	_timer2_instance->isr();
}

static FishinoTimerClass *_timer3_instance = NULL;
void __attribute__((interrupt(), nomips16)) _timer3InterruptHandler(void)
{
	_timer3_instance->isr();
}

static FishinoTimerClass *_timer4_instance = NULL;
void __attribute__((interrupt(), nomips16)) _timer4InterruptHandler(void)
{
	_timer4_instance->isr();
}

static FishinoTimerClass *_timer5_instance = NULL;
void __attribute__((interrupt(), nomips16)) _timer5InterruptHandler(void)
{
	_timer5_instance->isr();
}

static FishinoTimerClass *_timer23_instance = NULL;
void __attribute__((interrupt(), nomips16)) _timer23InterruptHandler(void)
{
	_timer23_instance->isr();
}

static FishinoTimerClass *_timer45_instance = NULL;
void __attribute__((interrupt(), nomips16)) _timer45InterruptHandler(void)
{
	_timer45_instance->isr();
}

///////////////////////////////////////////////////////////////////////////////////////

// internal interrupt handler
void FishinoTimerClass::isr(void)
{
	// if we've set an handler, just call it
	if(_interruptHandler)
		_interruptHandler(_handlerParameter);
	
	// clear interrupt flag on exit
	clearIntFlag(_irq);
}

// constructor
FishinoTimerClass::FishinoTimerClass()
{
	// Constructor code here
	_tcon = NULL;
	_pr = NULL;
	_vec = 0;
	_irq = 0;
	_ipl = 0;
	_spl = 0;
	_tmr = NULL;
	_ps_size = 2;
	
	_isrHandler = NULL;
	_interruptHandler = NULL;
	_interruptsEnabled = false;
}

// destructor
FishinoTimerClass::~FishinoTimerClass()
{
	// just in case...
	disableInterrupts();
}

// set timer frequenency
// @@@@ STILL TO FIX !!!!
bool FishinoTimerClass::setFrequency(uint32_t f)
{
	uint32_t f_pb = getPeripheralClock();
	uint32_t baseclock = f_pb;
	uint8_t ps = 0;

	if (_ps_size == 2)
	{
		if (baseclock / f > 65535)
		{
			baseclock = f_pb / 8;
			ps = 1;
		}

		if (baseclock / f > 65535)
		{
			baseclock = f_pb / 64;
			ps = 2;
		}

		if (baseclock / f > 65535)
		{
			baseclock = f_pb / 256;
			ps = 3;
		}
		_tcon->tckps = ps;
		_tcon->t32 = 0;
		*_pr = baseclock / f;
	}
	else if (_ps_size == 3)
	{
		if (baseclock / f > 65535)
		{
			baseclock = f_pb / 2;
			ps = 1;
		}

		if (baseclock / f > 65535)
		{
			baseclock = f_pb / 4;
			ps = 2;
		}

		if (baseclock / f > 65535)
		{
			baseclock = f_pb / 8;
			ps = 3;
		}

		if (baseclock / f > 65535)
		{
			baseclock = f_pb / 16;
			ps = 4;
		}

		if (baseclock / f > 65535)
		{
			baseclock = f_pb / 32;
			ps = 5;
		}

		if (baseclock / f > 65535)
		{
			baseclock = f_pb / 64;
			ps = 6;
		}

		if (baseclock / f > 65535)
		{
			baseclock = f_pb / 256;
			ps = 7;
		}
		_tcon->tckps = ps;
		_tcon->t32 = 0;
		*_pr = baseclock / f;
	}
	else
	{
	_tcon->tckps = 0;
	_tcon->t32 = 1;
	*_pr = (baseclock / f);
	}
	
	return true;
}

// start the timer
void FishinoTimerClass::start()
{
	_tcon->on = 1;
}

// stop the timer
void FishinoTimerClass::stop()
{
	_tcon->on = 0;
}

// attach an interrupt handler to timer
void FishinoTimerClass::_attachInterrupt(FISHINOTIMER_INTERRUPT_HANDLER isr, void *param)
{
	// if interrupts enabled, disable them
	if(_interruptsEnabled)
		disableInterrupts();
	
	// store user's interrupt handler
	_handlerParameter = param;
	_interruptHandler = isr;
	
	// enable interrupts again
	enableInterrupts();
}

// detach the interrupt handler from timer
void FishinoTimerClass::detachInterrupt()
{
	// if interrupts enabled, disable them
	if(_interruptsEnabled)
		disableInterrupts();
	
	// remove user's interrupt handler
	_interruptHandler = NULL;
	
}

// set interrupt priority
void FishinoTimerClass::setInterruptPriority(int pri)
{
	// store enabled flag -- we may want change priority
	// on a running timer
	bool en = _interruptsEnabled;

	// if interrupts enabled, disable them
	if(en)
		disableInterrupts();
	
	// store new priority
	_ipl = pri;
	
	// re-enable interrupts if needed
	if(en)
		enableInterrupts();
}
		
// enable interrupts
void FishinoTimerClass::enableInterrupts(void)
{
	// if we don't have any handler or already enabled, just do nothing
	if(!_interruptHandler || _interruptsEnabled)
		return;
	
	// connect external ISR handler
	setIntVector(_vec, _isrHandler);
	
	// setup interrupt priority
	setIntPriority(_vec, _ipl, _spl);
	
	// clear any pending interrupt
	clearIntFlag(_irq);
	
	// enable interrupts
	setIntEnable(_irq);
	
	// mark interrupts as enabled
	_interruptsEnabled = true;
}

// disable interrupts
void FishinoTimerClass::disableInterrupts(void)
{
	// if not enabled, just do nothing
	if(!_interruptsEnabled)
		return;
	
	// disable interrupts and vector
	clearIntEnable(_irq);
	setIntPriority(_vec, 0, 0);
	clearIntVector(_vec);
	
	// mark interrupts as disabled
	_interruptsEnabled = false;
}

// set timer's clock source
void FishinoTimerClass::setClockSource(FishinoTimerSource src)
{
	switch (src)
	{
		case FISHINOTIMER_SOURCE_INTERNAL:
			_tcon->tcs = 0;
			break;
		case FISHINOTIMER_SOURCE_EXTERNAL:
			_tcon->tcs = 1;
			break;
	}
}

// set timer's prescaler
bool FishinoTimerClass::setPrescaler(FishinoTimerPrescale ps)
{
	if(_ps_size == 2)
	{
		switch(ps)
		{
			case FISHINOTIMER_PRESCALE_1:
				_tcon->tckps = 0;
				break;
			case FISHINOTIMER_PRESCALE_8:
				_tcon->tckps = 1;
				break;
			case FISHINOTIMER_PRESCALE_64:
				_tcon->tckps = 2;
				break;
			case FISHINOTIMER_PRESCALE_256:
				_tcon->tckps = 3;
				break;
				
			// we shall maybe signal an error here...
			default:
				return false;
		}
	}
	else
	{
		_tcon->tckps = ps;
	}
	return true;
}

bool FishinoTimerClass::setPrescaler(int ps)
{
	switch(ps)
	{
		case 1:
			return setPrescaler(FISHINOTIMER_PRESCALE_1);
		case 2:
			return setPrescaler(FISHINOTIMER_PRESCALE_2);
		case 4:
			return setPrescaler(FISHINOTIMER_PRESCALE_4);
		case 8:
			return setPrescaler(FISHINOTIMER_PRESCALE_8);
		case 16:
			return setPrescaler(FISHINOTIMER_PRESCALE_16);
		case 32:
			return setPrescaler(FISHINOTIMER_PRESCALE_32);
		case 64:
			return setPrescaler(FISHINOTIMER_PRESCALE_64);
		case 256:
			return setPrescaler(FISHINOTIMER_PRESCALE_256);
			
		default:
			return false;
	}
}


// set timer's period in microseconds
void FishinoTimerClass::setPeriod(uint32_t per)
{
	*_pr = per;
}

// set timer's period in microseconds
bool FishinoTimerClass::setPeriodMicroseconds(uint32_t us)
{
	// resolution is 1 uSecond, not less
	if(!us)
		us = 1;
	
	uint64_t per = (uint64_t)getPeripheralClock() * us;
	per /= 1000000;
	uint32_t div = 1;
	if(_tcon->t32)
	{
		while(per > (uint32_t)-1)
		{
			per >>= 1;
			div <<= 1;
		}
	}
	else
	{
		while(per > (uint16_t)-1)
		{
			per >>= 1;
			div <<= 1;
		}
	}

	// check if can't cope with this period
	if(div > 256)
		return false;
	
	// correct prescale for 2 bit-sized prescalers
	if(_ps_size == 2)
	{
		// can use 1, 8, 64 and 256
		if(div == 2)
		{
			// prescale to 8
			div <<=  2;
			per >>= 2;
		}
		else if(div == 4)
		{
			// prescale to 8
			div <<=  1;
			per >>= 1;
		}
		else if(div == 16)
		{
			// prescale to 64
			div <<=  2;
			per >>= 2;
		}
		else if(div == 32)
		{
			// prescale to 64
			div <<=  1;
			per >>= 1;
		}
		else if(div == 128)
		{
			// prescale to 256
			div <<=  1;
			per >>= 1;
		}
	}
	else
	{
		// we need to check also 3 bit prescaler, 128 is not allowed
		if(div == 128)
		{
			// prescale to 256
			div <<=  1;
			per >>= 1;
		}
	}
	
	// try to set prescaler
	if(!setPrescaler(div))
		return false;
	
	// set the period
	*_pr = per;

	// all ok
	return true;
}

// set timer's period in milliseconds
bool FishinoTimerClass::setPeriodMilliseconds(uint32_t ms)
{
	return setPeriodMicroseconds(ms * 1000);
}

// reset timer
void FishinoTimerClass::reset()
{
	*_tmr = 0;
}

// get current timer ticks
uint32_t FishinoTimerClass::getCount()
{
	return *_tmr;
}

// get current timer ticks and reset it
uint32_t FishinoTimerClass::getAndResetCount()
{
	uint32_t t = *_tmr;
	*_tmr = 0;
	return t;
}

void FishinoTimerClass::enableGate()
{
	_tcon->tgate = 1;
}

void FishinoTimerClass::disableGate()
{
	_tcon->tgate = 0;
}

// get a Register Value
uint32_t FishinoTimerClass::getRegisters(uint32_t *addr)
{
	return (uint32_t) *addr;
}


///////////////////////////////////////////////////////////////////////////////////////

struct _FishinoTimer_Initializer
{
	int _vec;
	int _irq;
	int _ipl;
	int _spl;
	volatile void *_tcon;
	volatile void *_pr;
	int _ps_size;
	volatile void *_tmr;
	void (*isrHandler)(void);
	FishinoTimerClass **_instance;
};

const _FishinoTimer_Initializer _FishinoTimer_Initializers[] =
{
	{ _TIMER_1_VECTOR, _TIMER_1_IRQ, 4, 0, &T1CON, &PR1, 2, &TMR1, _timer1InterruptHandler, &_timer1_instance },
	{ _TIMER_2_VECTOR, _TIMER_2_IRQ, 4, 0, &T2CON, &PR2, 3, &TMR2, _timer2InterruptHandler, &_timer2_instance },
	{ _TIMER_3_VECTOR, _TIMER_3_IRQ, 4, 0, &T3CON, &PR3, 3, &TMR3, _timer3InterruptHandler, &_timer3_instance },
	{ _TIMER_4_VECTOR, _TIMER_4_IRQ, 4, 0, &T4CON, &PR4, 3, &TMR4, _timer4InterruptHandler, &_timer4_instance },
	{ _TIMER_5_VECTOR, _TIMER_5_IRQ, 4, 0, &T5CON, &PR5, 3, &TMR5, _timer5InterruptHandler, &_timer5_instance },
	{ _TIMER_3_VECTOR, _TIMER_3_IRQ, 4, 0, &T2CON, &PR2, 4, &TMR2, _timer23InterruptHandler, &_timer23_instance },
	{ _TIMER_5_VECTOR, _TIMER_5_IRQ, 4, 0, &T4CON, &PR4, 4, &TMR4, _timer45InterruptHandler, &_timer45_instance }
};

void _FishinoTimer_Initialize(FishinoTimerClass &t, uint8_t index)
{
	_FishinoTimer_Initializer const &i = _FishinoTimer_Initializers[index];
	
	t._vec = i._vec;
	t._irq = i._irq;
	t._ipl = i._ipl;
	t._spl = i._spl;
	t._tcon = (tcon_s *)i._tcon;
	t._pr = (volatile uint32_t *)i._pr;
	t._ps_size = i._ps_size;
	if(i._ps_size == 4)
		t._tcon->t32 = 1;
	else
		t._tcon->t32 = 0;
	t._tmr = (volatile uint32_t *)i._tmr;
	t._isrHandler = i.isrHandler;
	
	*i._instance = &t;
}

#define FISHINOTIMER_DEF(name, idx) \
FishinoTimerClass &_timer##name() \
{ \
	static FishinoTimerClass t; \
	static bool initialized = false; \
	if(!initialized) \
	{ \
		_FishinoTimer_Initialize(t, idx); \
		initialized = true; \
	} \
	return t; \
}

FISHINOTIMER_DEF(1, 0)
FISHINOTIMER_DEF(2, 1)
FISHINOTIMER_DEF(3, 2)
FISHINOTIMER_DEF(4, 3)
FISHINOTIMER_DEF(5, 4)
FISHINOTIMER_DEF(23, 5)
FISHINOTIMER_DEF(45, 6)

#undef FISHINOTIMER_DEF

#endif
