//////////////////////////////////////////////////////////////////////////////////////
//						 		ClosedLoop.cpp										//
//																					//
//			Library to handle MotorFish closed loop stepper driver board			//
//																					//
//		Copyright (c) 2019 Massimo Del Fedele.  All rights reserved.				//
//																					//
//	Redistribution and use in source and binary forms, with or without				//
//	modification, are permitted provided that the following conditions are met:		//
//																					//
//	- Redistributions of source code must retain the above copyright notice,		//
//	  this list of conditions and the following disclaimer.							//
//	- Redistributions in binary form must reproduce the above copyright notice,		//
//	  this list of conditions and the following disclaimer in the documentation		//
//	  and/or other materials provided with the distribution.						//
//																					//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"		//
//	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE		//
//	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE		//
//	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE		//
//	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR				//
//	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF			//
//	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS		//
//	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN			//
//	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)			//
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE		//
//	POSSIBILITY OF SUCH DAMAGE.														//
//																					//
//  Version 7.8.0 -- March 2019		Initial version									//
//	Version 8.0.0 - 26/07/2020 - UPDATED FOR FIRMWARE 8.0.0							//
//																					//
//////////////////////////////////////////////////////////////////////////////////////
#include "MotorFish.h"

#include <FishinoTimer.h>

#define DEBUG_LEVEL_INFO
#include <FishinoDebug.h>

// time between sync, in microseconds
volatile uint32_t ClosedLoopClass::_syncTime = 50;

// syncing active flag
volatile bool ClosedLoopClass::_syncing = false;
		
// enabled flag
volatile bool ClosedLoopClass::_enabled = false;
		
// the PID variables (-32768..+32767)
volatile int16_t ClosedLoopClass::_Kp = 16384;
volatile int16_t ClosedLoopClass::_Kd = 0;
volatile int16_t ClosedLoopClass::_Ki = 0;

// integral factor damping
// used as integralDamp / 32767 to reduce effects of farthest integral terms
// should be near unity (32767)
volatile int32_t ClosedLoopClass::_integralDamp = 32000;

// running pid variables
volatile int32_t ClosedLoopClass::_prevDiff = 0;
volatile int32_t ClosedLoopClass::_integralTerm = 0;

// the required motor position
volatile int32_t ClosedLoopClass::_requiredPosition = 0;

// the sync timed routine
void ClosedLoopClass::_syncISR(void)
{
	static bool inside = false;
	if(inside)
		return;
	inside = true;
	
	// get current position in STEP coordinates
	int32_t currentPos = Encoder.motorPos();
	
	// calculate position difference
	int32_t diffPos = _requiredPosition - currentPos;
	
//	DEBUG_INFO("REQ:%06d - ENC:%06d\n", _requiredPosition, currentPos);
	
	// find shaft position (0..360 --> 0..200 * 64)
	int32_t shaftPos = currentPos % (200 * 64);
	if(shaftPos < 0)
		shaftPos += 200 * 64;
	
	// we calculate PID value as a SIGNED current
	// then we choose direction basing on sign

	// update integral term and clamp it
	_integralTerm = (_integralDamp * _integralTerm) / 32767 + diffPos;
	if(_integralTerm > 65535)
		_integralTerm = 65535;
	else if(_integralTerm < -65535)
		_integralTerm = -65535;
	
	// get derivative term and clamp it
	int32_t derivTerm = diffPos - _prevDiff;
	if(derivTerm > 65535)
		derivTerm = 65535;
	else if(derivTerm < -65535)
		derivTerm = -65535;
	
	// clamp difference too
	if(diffPos > 65535)
		diffPos = 65535;
	else if(diffPos < -65535)
		diffPos = -65535;

	// update previous difference
	_prevDiff = diffPos;
	
	// apply proportional, integral and derivative terms
	int64_t current = (_Kp * (int64_t)diffPos + _Ki * (int64_t)_integralTerm + _Kd * (int64_t)derivTerm) / 64;

	// apply shaft direction from current sign
	if(diffPos < 0)
		diffPos = -diffPos;
	int32_t nextPhase;

/*
	if(current > 0)
	{
		if(diffPos > 62)
			nextPhase = shaftPos + 64;
		else
			nextPhase = shaftPos + diffPos + 2;
	}
	else
	{
		current = -current;
		if(diffPos > 62)
			nextPhase = shaftPos - 64;
		else
			nextPhase = shaftPos - diffPos - 2;
	}
*/
	if(current > 0)
		nextPhase = shaftPos + 64;
	else
	{
		current = -current;
		nextPhase = shaftPos - 64;
	}

	
	nextPhase %= 256;
	if(nextPhase < 0)
		nextPhase += 256;

	// clamp current
	if(current > 65535)
		current = 65535;

	// set motor current and phase
	Stepper.setPhaseFactor(nextPhase, current);

	inside = false;
}

// constructor
ClosedLoopClass::ClosedLoopClass()
{
	// not syncing
	_syncing = false;
	
	// 50 uSec default sync time
	_syncTime = 50;

	// not enabled
	_enabled = false;
	
	// required position at zero
	_requiredPosition = 0;
	
	// we need a stable reference between shaft position
	// (on an integral step) and motor position 0
	// so we energize the stepper, slowly, before resetting positions
	for(uint32_t i = 255; i <= 65535; i += 256)
	{
		Stepper.setPhaseFactor(0, i);
		Stepper.energize(true);
		delay(1);
	}
	
	// reset encoder position
	Encoder.resetMotorPos();
	
	// reset required position
	_requiredPosition = 0;

	// de-energize the motor
	Stepper.energize(false);
}

// destructor
ClosedLoopClass::~ClosedLoopClass()
{
	// just im case, de-energize stepper motor
	Stepper.energize(false);
}

// set individual PID parameters
void ClosedLoopClass::setKp(int16_t kp)
{
	setPID(kp, _Ki, _Kd);
}

void ClosedLoopClass::setKi(int16_t ki)
{
	setPID(_Kp, ki, _Kd);
}

void ClosedLoopClass::setKd(int16_t kd)
{
	setPID(_Kp, _Ki, kd);
}

// set PID parameters
void ClosedLoopClass::setPID(int16_t kp, int16_t ki, int16_t kd)
{
	bool s = _syncing;
	stopSync();

	_Kp = kp;
	_Ki = ki;
	_Kd = kd;
	
	if(s)
		startSync();
}

// set the integral damping factor
void ClosedLoopClass::setIntegralDamp(int32_t d)
{
	bool s = _syncing;
	stopSync();

	_integralDamp = d;
	
	if(s)
		startSync();
}

// set sync time (microseconds)
void ClosedLoopClass::setSyncTime(uint32_t tim)
{
	bool s = _syncing;
	stopSync();
	
	_syncTime = tim;

	if(s)
		startSync();
}

// start syncing position
void ClosedLoopClass::startSync(void)
{
	if(_syncing)
		return;

	// reset running PID variables
	_integralTerm = 0;
	_prevDiff = 0;
	
	// use Timer5
	Timer5.attachInterrupt(_syncISR);
	Timer5.setPeriodMicroseconds(_syncTime);
	Timer5.start();
	
	_syncing = true;
}

// stop syncing position
void ClosedLoopClass::stopSync(void)
{
	if(!_syncing)
		return;
	
	// stop timed syncing
	Timer5.stop();
	
	_syncing = false;
}

// enable/disable the motor
void ClosedLoopClass::enable(bool en)
{
	if(_enabled == en)
		return;

	_enabled = en;

	// disable sincing temporarly
	stopSync();
	
	if(!en)
	{
		Stepper.energize(false);
		return;
	}
	
	// by now we just reset position on re-enable

	// ramp-up the motor slowly, to have a good reference point
	for(uint32_t i = 255; i <= 65535; i += 256)
	{
		Stepper.setPhaseFactor(0, i);
		Stepper.energize(true);
		delay(1);
	}

	// reset encoder position
	Encoder.resetMotorPos();
	
	// reset required position
	_requiredPosition = 0;

	// re-start syncing
	startSync();
}

// execute a step
void ClosedLoopClass::step(bool dir)
{
	_requiredPosition += (dir ? 1 : -1);
}


ClosedLoopClass &__getClosedLoopClass(void)
{
	static ClosedLoopClass closedLoop;
	return closedLoop;
}
