//////////////////////////////////////////////////////////////////////////////////////
//						 		Encoder.cpp											//
//																					//
//			Library to handle MotorFish closed loop stepper driver board			//
//																					//
//		Copyright (c) 2019 Massimo Del Fedele.  All rights reserved.				//
//																					//
//	Redistribution and use in source and binary forms, with or without				//
//	modification, are permitted provided that the following conditions are met:		//
//																					//
//	- Redistributions of source code must retain the above copyright notice,		//
//	  this list of conditions and the following disclaimer.							//
//	- Redistributions in binary form must reproduce the above copyright notice,		//
//	  this list of conditions and the following disclaimer in the documentation		//
//	  and/or other materials provided with the distribution.						//
//																					//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"		//
//	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE		//
//	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE		//
//	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE		//
//	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR				//
//	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF			//
//	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS		//
//	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN			//
//	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)			//
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE		//
//	POSSIBILITY OF SUCH DAMAGE.														//
//																					//
//  Version 7.8.0 -- March 2019		Initial version									//
//	Version 8.0.0 - 26/07/2020 - UPDATED FOR FIRMWARE 8.0.0							//
//																					//
//////////////////////////////////////////////////////////////////////////////////////
#include "Encoder.h"

#include <Wire.h>
#include <FishinoEEPROM.h>

#define DEBUG_LEVEL_INFO
#include <FishinoDebug.h>

#define SDA_INT	44
#define SCL_INT 45

// multiple NOPs to create short delays
template<int N> inline void nop()
{
    nop<N-1>();
    asm("nop");
}

template<> inline void nop<0>() { }

//////////////////////////////////////////////////////////////////
//				MotorFish encoder connections					//
//////////////////////////////////////////////////////////////////

const int CLOCK		= 31;
const int DATA		= 32;
const int ZERO		= 33;
const int ERROR		= 34;
const int MAG		= 35;
const int A			= 36;
const int B			= 37;
const int RI		= 38;
const int VOUT_TOUT	= 39;
const int TD_PCOS	= 40;
const int W_NCOS	= 41;
const int V_PSIN	= 42;
const int U_NSIN	= 43;

// fast I/O on CLOCK and DATA pins
// used by FAST_SSI routines
#define CLOCK_PORT	PORTD
#define CLOCK_BIT	2
#define DATA_PORT	PORTD
#define DATA_BIT	11
#define WRITE_CLOCK(x) if(x) CLOCK_PORT |= _BV(CLOCK_BIT); else CLOCK_PORT &= ~_BV(CLOCK_BIT)
#define READ_DATA ((DATA_PORT & _BV(DATA_BIT)) ? 1 : 0)

#define ENCODER_WIRE		Wire0
#define	ENCODER_ADDRESS		0x00

// SPI settings used to read SSI position
#ifndef USE_FAST_SSI
	SPISettings EncoderClass::_spiSettings;
#endif

	// as it's not safe to print errors from inside ISR, we use
	// some flags that can be queried by main application
	volatile uint16_t EncoderClass::_errorFlags;
	volatile uint16_t EncoderClass::_errorSSI;
	volatile uint32_t EncoderClass::_errorPrevMPos;
	volatile uint32_t EncoderClass::_errorCurMPos;
	volatile uint16_t EncoderClass::_errorMPOS_QUAD;
	volatile uint16_t EncoderClass::_error_MPOS_SSI;

#ifdef USE_QUADRATURE_SIGNALS

	// quadrature reading value
	volatile uint8_t EncoderClass::_quadratureData = 0;
	
	// lookup table for quadrature encoder reading
	const int8_t quadratureLookupTable[] = {
		0, -1, 1, 0, 1, 0, 0, -1, -1, 0, 0, 1, 0, 1, -1, 0
	};
	
#endif

// the encoder placement (true = backside, false = front side)
volatile bool EncoderClass::_placement = true;
		
// the encoder positive direction (true = CW, false = CCW)
// this is stored to allow encoder placement change
volatile bool EncoderClass::_direction = true;
		
// enable/disable calibration
volatile bool EncoderClass::_calibrationEnabled = false;

// the absolute motor position
volatile int32_t EncoderClass::_motorPos = 0;

// previous SSI reading used to keep motorPos updater
volatile int32_t EncoderClass::_prevSSIPos = 0;

// last hardware-read SSI position
volatile uint16_t EncoderClass::_lastReadSSIPos = 0;

// SSI reading for motor 0 position
volatile uint16_t EncoderClass::_zeroPos = 0;

// calibration data - kept in EEPROM with direct access
// data is inside first 4096 eeprom bytes
bool EncoderClass::_hasCalibrationData = false;
extern uint8_t _eeprom_start;
int8_t * EncoderClass::_calibrationData = (int8_t *)&_eeprom_start;

// eeprom mode for I2C
bool EncoderClass::_eepromMode = false;

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

// read encoder register
uint16_t EncoderClass::readReg(uint8_t reg)
{
	// try 5 times in case of error
	for(int i = 0; i < 5; i++)
	{
// DEBUG_INFO_N("READ %d ", i);
		if(!_eepromMode && reg <= 3)
			reg += 48;
		
		uint16_t res = 0;
		
		ENCODER_WIRE.beginTransmission(ENCODER_ADDRESS);
// DEBUG_INFO_N("A");
		ENCODER_WIRE.write(reg);
// DEBUG_INFO_N("B");
		
		// if failed terminate current transmission
		// and retry
		if(ENCODER_WIRE.endTransmission(false))
		{
// DEBUG_INFO_N("C");
			ENCODER_WIRE.endTransmission(true);
// DEBUG_INFO_N("D");
			delay(1);
			continue;
		}
// DEBUG_INFO_N("E");
		ENCODER_WIRE.requestFrom(ENCODER_ADDRESS, 2);
// DEBUG_INFO_N("F");
		res = ENCODER_WIRE.read();
// DEBUG_INFO_N("G");
		res = (res << 8) | ENCODER_WIRE.read();
// DEBUG_INFO_N("H");
// DEBUG_INFO_N("DONE\n");
		return res;
	}
// DEBUG_INFO_N("FAILED\n");
	return 0;
}

// write encoder register
void EncoderClass::writeReg(uint8_t reg, uint16_t data)
{
	if(!_eepromMode && reg <= 3)
		reg += 48;

	// try 5 times in case of error
	for(int i = 0; i < 5; i++)
	{
// DEBUG_INFO_N("WRITE %d ", i);
		ENCODER_WIRE.beginTransmission(ENCODER_ADDRESS);
		ENCODER_WIRE.write(reg);
		ENCODER_WIRE.write((uint8_t)(data >> 8));
		ENCODER_WIRE.write((uint8_t)data);
		if(ENCODER_WIRE.endTransmission())
		{
			delay(1);
			continue;
		}
// DEBUG_INFO_N("DONE\n");
		break;
	}
// DEBUG_INFO_N("FAILED\n");
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
// 											set/get some encoder registers									//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
		
// interpolator power
void EncoderClass::enableInterpolator(bool en)
{
	uint16_t r = readReg(0x00);
	if(en)
		r &= ~_BV(15);
	else
		r |= _BV(15);
	writeReg(0x00, r);
}

bool EncoderClass::isInterpolatorEnabled(void)
{
	uint16_t r = readReg(0x00);
	return !(r & _BV(15));
}

// AGC
void EncoderClass::enableAGC(bool en)
{
	uint16_t r = readReg(0x00);
	if(en)
		r &= ~_BV(14);
	else
		r |= _BV(14);
	writeReg(0x00, r);
}

bool EncoderClass::isAGCEnabled(void)
{
	uint16_t r = readReg(0x00);
	return !(r & _BV(14));
}

// interpolator delay
void EncoderClass::enableInterpolatorDelay(bool en)
{
	uint16_t r = readReg(0x00);
	if(en)
		r &= ~_BV(12);
	else
		r |= _BV(12);
	writeReg(0x00, r);
}

bool EncoderClass::isInterpolatorDelayEnabled(void)
{
	uint16_t r = readReg(0x00);
	return ~(r & _BV(12));
}

// power down rate
void EncoderClass::setPowerDownRate(PDN_RATES rate)
{
	uint16_t r = readReg(0x00);
	r &= ~(_BV(11) | _BV(10));
	r |= ((uint16_t)rate) << 10;
	writeReg(0x00, r);
}

EncoderClass::PDN_RATES EncoderClass::getPowerDownRate(void)
{
	uint16_t r = readReg(0x00);
	r = (r >> 10) & 0x03;
	return (PDN_RATES)r;
}

// power down state
void EncoderClass::enablePowerDown(bool en)
{
	uint16_t r = readReg(0x00);
	if(en)
		r &= ~_BV(9);
	else
		r |= _BV(9);
	writeReg(0x00, r);
}

bool EncoderClass::isPowerDownEnabled(void)
{
	uint16_t r = readReg(0x00);
	return ~(r & _BV(9));
}

// regulator voltage
void EncoderClass::setRegulatorVoltage(REG_VOLTAGES v)
{
	uint16_t r = readReg(0x00);
	if(v == RV_3)
		r &= ~_BV(8);
	else
		r |= _BV(8);
	writeReg(0x00, r);
}

EncoderClass::REG_VOLTAGES EncoderClass::getRegulatorVoltage(void)
{
	uint16_t r = readReg(0x00);
	if(r & _BV(8))
		return RV_3_3;
	else
		return RV_3;
}

// device address
void EncoderClass::setDeviceAddress(uint8_t addr)
{
	uint16_t r = readReg(0x00) & 0xff80;
	r |= (addr & 0x7f);
	writeReg(0x00, r);
}

uint8_t EncoderClass::getDeviceAddress(void)
{
	uint16_t r = readReg(0x00);
	return (uint8_t)(r & 0x7f);
}

// A/B/Ri outputs
void EncoderClass::enableABRIOutputs(bool en)
{
	uint16_t r = readReg(0x01);
	if(en)
		r &= ~_BV(15);
	else
		r |= _BV(15);
	writeReg(0x01, r);
}

bool EncoderClass::isABRIOutputsEnabled(void)
{
	uint16_t r = readReg(0x01);
	return !(r & _BV(15));
}

// UVW output type
void EncoderClass::setUVWOutputType(UVW_SEL sel)
{
	uint16_t r = readReg(0x01);
	if(sel == UVW_TACHO)
		r &= ~_BV(14);
	else
		r |= _BV(14);
	writeReg(0x01, r);
}

EncoderClass::UVW_SEL EncoderClass::getUVWOutputType(void)
{
	uint16_t r = readReg(0x01);
	if(r & _BV(14))
		return SINE_DIFF;
	else
		return UVW_TACHO;
}

// error output type
void EncoderClass::setErrorOutputType(ERR_PIN_SEL sel)
{
	uint16_t r = readReg(0x01);
	if(sel == ERR_SIGNAL)
		r &= ~_BV(13);
	else
		r |= _BV(13);
	writeReg(0x01, r);
}

EncoderClass::ERR_PIN_SEL EncoderClass::getErrorOutputType(void)
{
	uint16_t r = readReg(0x01);
	if(r & _BV(13))
		return ERR_AMPLITUDE;
	else
		return ERR_SIGNAL;
}

// output direction - encoder placement INDEPENDENT
void EncoderClass::setOutputDirection0(OUTPUT_DIR dir)
{
	uint16_t r = readReg(0x01);
	if(dir == DIR_POS)
		r &= ~_BV(12);
	else
		r |= _BV(12);
	writeReg(0x01, r);
}

EncoderClass::OUTPUT_DIR EncoderClass::getOutputDirection0(void)
{
	uint16_t r = readReg(0x01);
	if(r & _BV(12))
		return DIR_NEG;
	else
		return DIR_POS;
}

// zero position
void EncoderClass::setZeroPosition(uint16_t pos)
{
	uint16_t r = readReg(0x01);
	r &= 0xf000;
	r |= pos & 0x0fff;
	writeReg(0x01, r);
}

uint16_t EncoderClass::getZeroPosition(void)
{
	return readReg(0x01) & 0x0fff;
}

// output position type
void EncoderClass::setOutputPositionType(OUTPUT_POS_TYPE type)
{
	uint16_t r = readReg(0x02);
	if(type == POS_RELATIVE)
		r &= ~_BV(7);
	else
		r |= _BV(7);
	writeReg(0x02, r);
}

EncoderClass::OUTPUT_POS_TYPE EncoderClass::getOutputPositionType(void)
{
	uint16_t r = readReg(0x02);
	if(r & _BV(7))
		return POS_ABSOLUTE;
	else
		return POS_RELATIVE;
}

// digital hysteresis
void EncoderClass::setDigitalHysteresis(uint8_t h)
{
	uint16_t r = readReg(0x02);
	r &= 0xff80;
	r |= h & 0x7f;
	writeReg(0x02, r);
}

uint8_t EncoderClass::getDigitalHysteresis(void)
{
	return (uint8_t)(readReg(0x02) & 0x7f);
}

// Vout/Tout ooutput type
void EncoderClass::setVoutToutOutputType(VOUT_TOUT_SEL sel)
{
	uint16_t r = readReg(0x03);
	if(sel == VOUT_TOUT_POS)
		r &= ~_BV(15);
	else
		r |= _BV(15);
	writeReg(0x03, r);
}

EncoderClass::VOUT_TOUT_SEL EncoderClass::getVoutToutOutputType(void)
{
	uint16_t r = readReg(0x03);
	if(r & _BV(15))
		return VOUT_TOUT_TACHO;
	else
		return VOUT_TOUT_POS;
}

// linear voltage period
void EncoderClass::setLinearVoltagePeriod(LINEAR_VOLTAGE_PERIOD period)
{
	uint16_t r = readReg(0x03) & 0x9fff;
	r |= ((uint16_t)period) << 13;
	writeReg(0x03, r);
}

EncoderClass::LINEAR_VOLTAGE_PERIOD EncoderClass::getLinearVoltagePeriod(void)
{
	return (LINEAR_VOLTAGE_PERIOD)((readReg(0x03) >> 13) & 0x03);
}

// SSI output setting
void EncoderClass::setSSISettings(SSI_SETTINGS s)
{
	uint16_t r = readReg(0x03) & 0xe7ff;
	r |= ((uint16_t)s) << 11;
	writeReg(0x03, r);
}

EncoderClass::SSI_SETTINGS EncoderClass::getSSISettings(void)
{
	return (SSI_SETTINGS)((readReg(0x03) >> 11) & 0x03);
}

// Tacho range
void EncoderClass::setTachoRange(TACHO_RANGES range)
{
	uint16_t r = readReg(0x03) & 0xfe3f;
	r |= ((uint16_t)range) << 6;
	writeReg(0x03, r);
}

EncoderClass::TACHO_RANGES EncoderClass::getTachoRange(void)
{
	return (TACHO_RANGES)((readReg(0x03) >> 6) & 0x07);
}

// UVW period
void EncoderClass::setUVWPeriod(UVW_PERIODS period)
{
	uint16_t r = readReg(0x03) & 0xffc7;
	r |= ((uint16_t)period) << 3;
	writeReg(0x03, r);
}

EncoderClass::UVW_PERIODS EncoderClass::getUVWPeriod(void)
{
	return (UVW_PERIODS)((readReg(0x03) >> 3) & 0x07);
}

// interpolator factor rate
void EncoderClass::setInterpolatorRate(INTERP_RATE rate)
{
	uint16_t r = readReg(0x03) & 0xfff8;
	r |= ((uint16_t)rate);
	writeReg(0x03, r);
}

EncoderClass::INTERP_RATE EncoderClass::getInterpolatorRate(void)
{
	return (INTERP_RATE)(readReg(0x03) & 0x07);
}

// check if position is valid
bool EncoderClass::isRelativePositionValid(void)
{
	return readReg(32) & _BV(15);
}

bool EncoderClass::isAbsolutePositionValid(void)
{
	return readReg(33) & _BV(15);
}

// read absolute and relative position
uint16_t EncoderClass::readRelativePosition(void)
{
	return readReg(32) & 0x0fff;
}

uint16_t EncoderClass::readAbsolutePosition(void)
{
	return readReg(33) & 0x0fff;
}

// check magnet status
bool EncoderClass::isMagnetTooFar(void)
{
	return readReg(34) & _BV(14);
}

bool EncoderClass::isMagnetTooClose(void)
{
	return readReg(34) & _BV(13);
}

bool EncoderClass::isMagnetOk(void)
{
	return !(readReg(34) & (_BV(13) | _BV(14)));
}

// tacho
bool EncoderClass::isTachoOverflow(void)
{
	return readReg(35) & _BV(10);
}

uint16_t EncoderClass::readTacho(void)
{
	return readReg(35) & 0x03ff;
}

// read/write eeprom (44 bytes)
void EncoderClass::writeEEProm(uint8_t reg, uint8_t data)
{
	if(reg > 43)
		return;
	uint8_t eReg = (reg >> 1) + 8;
	uint16_t r = readReg(eReg);
	if(reg & 1)
		r = (r & 0x00ff) | (((uint16_t)data) << 8);
	else
		r = (r & 0xff00) | data;
	writeReg(eReg, r);
}

uint8_t EncoderClass::readEEProm(uint8_t reg)
{
	if(reg > 43)
		return 0;
	uint8_t eReg = (reg >> 1) + 8;
	uint16_t data = readReg(eReg);
	if(reg & 1)
		data >>= 8;
	return (uint8_t)data;
}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef USE_QUADRATURE_SIGNALS

// interrupt handler used to read quadrature signals
void EncoderClass::_quadratureISR(void)
{
	// the 3 quadrature inputs (A == RB12, B == RB13 and RI == RB15) uses port B
	
	// read the change notification flags
	volatile uint32_t stat __attribute__((unused)) = CNSTATB;
	
	// read the port value
	volatile uint32_t port __attribute__((unused)) = PORTB;
	
	// clear port B change interrupt
	IFS1CLR = _IFS1_CNBIF_MASK;
	
	// clear the changes
	CNSTATBCLR = 0xffff;
	
	_quadratureData = (_quadratureData << 2) & 0b1100;
	_quadratureData |= ((port >> 12) & 0x03);
	_motorPos += quadratureLookupTable[_quadratureData];
}

#endif

// reset motor position
// (set to 0, re-read zero SSI position)
// DO NOT CALL this one from interrupt handler!!!
void EncoderClass::_resetMotorPos(void)
{
	// temporary disable quadrature interrupt
#ifdef USE_QUADRATURE_SIGNALS
	CNENB = 0;
	CNSTATB = 0;
#endif
	
	// set motor position to 0
	_motorPos = 0;
	
	// read SSI position corresponding to current pos
	// reading shall be calibrated, if calibration is available
	do
	{
		_zeroPos = _readSSIPosition();
	}
	while(_zeroPos == 0xffff);
	
//	DEBUG_INFO("Zero position is: %d\n", _zeroPos);
	
	// store previous SSI reading
	_prevSSIPos = _zeroPos;
	
	// re-enable quadrature interrupt
#ifdef USE_QUADRATURE_SIGNALS
	CNSTATB = 0;
	CNENB = _BV(12) | _BV(13);
#endif
}

// reads absolute position from encoder
// a 12 bit number from 0 to 4095
// low-level routine, no checks at all
// to be called ONLY by high level version
uint16_t EncoderClass::_readSSIPosition(void)
{
	static bool inside = false;
	
	// avoid intermixed calls
	if(inside)
		return 0xffff;
	inside = true;
	
	disableInterrupts();

	
#ifndef USE_FAST_SSI
	// normal SSI -- use SPI0 to read it
	// you can't read faster than tm time (~ 25 uSec)
	
	// check if SSI monoflop is ready to get data
	// otherwise just return -1
	if(!(DATA_PORT & _BV(DATA_BIT)))
	{
		enableInterrupts();
		inside = false;
		return 0xffff;
	}
	
	// read position as 2 bytes
	// first byte is MSB, with first bit not valid, so 7 bit valid
	// second byte is LSB, high part, 5 bit valid
	uint8_t hi = SPI0.transfer(0x00) & 0x7f;
	uint8_t lo = SPI0.transfer(0x00) & 0xf8;
	
	// combine to true data
	uint16_t ssiPos = (((uint16_t)hi) << 5) | (lo >> 3);

#else

	// fast SSI read -- we read JUST 12 bits of data
	// we must not wait for internal monoflop to reset
	// if we read faster than tm time
	// we use a software, real-time SPI routine
	// remember, PIC clock is 50 MHz
	uint16_t ssiPos = 0;
	WRITE_CLOCK(0);
	nop<5>();
	WRITE_CLOCK(1);
	for(int i = 0; i < 12; i++)
	{
		ssiPos <<= 1;
		nop<5>();
		ssiPos |= READ_DATA;
		WRITE_CLOCK(0);
		nop<5>();
		WRITE_CLOCK(1);
	}

#endif

	// release inside guard
	enableInterrupts();
	inside = false;

	// minimal error check
	if(ssiPos > 4095)
		return 0xffff;
	
	// calibrate it if requested
	if(_hasCalibrationData && _calibrationEnabled)
	{
		int32_t nPos = (int32_t)ssiPos + _calibrationData[ssiPos];
		if(nPos < 0)
			nPos += 4096;
		else if(nPos >= 4096)
			nPos -= 4096;
		ssiPos = nPos;
	}
	
	_lastReadSSIPos = ssiPos;
	return ssiPos;
}

// constructor
EncoderClass::EncoderClass()
{
	// check if we've got calibration data
	// we check first 100 bytes; if all 0xff, no data
	// otherwise we assume that device has been calibrated
	_hasCalibrationData = false;
	_calibrationEnabled = false;
	for(int i = 0; i < 100; i++)
	{
		if((uint8_t)_calibrationData[i] != 0xff)
		{
			_hasCalibrationData = true;
			_calibrationEnabled = true;
			break;
		}
	}

	// clear error flags and values
	clearErrors();

	// initialize I2C interface
	ENCODER_WIRE.begin();
	ENCODER_WIRE.setClock(400000);
	
	// setup regulator to 3v
	setRegulatorVoltage(RV_3);
	
	// enable AGC
	enableAGC();
	
	// enable interpolator
	enableInterpolator();
//	disableInterpolatorDelay();
	
#ifdef USE_QUADRATURE_SIGNALS
	// enable quadrature output
	enableABRIOutputs();
#else
	disableABRIOutputs();
#endif
	
	// disable differential sine outputs
	// otherwise interpolator may work uncorrectly
	setUVWOutputType(UVW_TACHO);
	
	// select error output on error pin
	setErrorOutputType(ERR_SIGNAL);
	
	// positive (CW) direction (can be changed later)
	// motor backside placement (can be changed later)
	_placement = true;
	_direction = true;
	setOutputDirection0(DIR_POS);

	// reset zero position, just in case
	// (we use anyways absolute, non-zeroed position
	setZeroPosition(0);
	
	// we want absolute position reading SSI
	setOutputPositionType(POS_ABSOLUTE);
	
	// set digital hysteresis
	// smaller = more precision but also many quadrature interrupts
	// @@ MUST CHECK THIS ONE!
#ifdef USE_QUADRATURE_SIGNALS
	setDigitalHysteresis(6);
#else
	setDigitalHysteresis(0);
#endif
	
	// get position data on Vout/Tout pins
	setVoutToutOutputType(VOUT_TOUT_POS);
	
	// linear voltage period on a full turn (360)
	setLinearVoltagePeriod(PERIOD_360);
	
	// setup SSI interface
#ifdef USE_FAST_SSI

	// configure SSI for single read
	// around 25 uSec needed between reads
	setSSISettings(SSI_RING_REFRESH);
	
#else
	// max frequency is by datasheet 4 MHz...
	// I've tested with 10 MHz and it still works, but...
	_spiSettings = SPISettings(4000000, MSBFIRST, SPI_MODE2);
	SPI0.begin();
	SPI0.beginTransaction(_spiSettings);

	// configure SSI for single read
	// around 25 uSec needed between reads
	setSSISettings(SSI_NO_RING);
	
#endif

	// setup tacho measuring range
	setTachoRange(TACHO_122880);
	
	// set number of UVW periods/turn
	setUVWPeriod(UVW_1);

	// set interpolator rate to max precision
	setInterpolatorRate(RATE_4096);

	// initialize SSI clock and data lines
	pinMode(CLOCK, OUTPUT);
	digitalWrite(CLOCK, HIGH);
	pinMode(DATA, INPUT);
	
	// give some time to setup
	delay(20);

	// setup interrupt routine to read quadrature signals
	// the 3 quadrature inputs (A == RB12, B == RB13 and RI == RB15) uses port B
#ifdef USE_QUADRATURE_SIGNALS
	TRISBSET = _BV(12) | _BV(13);
	ANSELBCLR = _BV(12) | _BV(13);
	CNCONBbits.ON = 1;
	CNPUB = 0;
	CNPDB = 0;
	CNENB = _BV(12) | _BV(13);
	CNSTATB = 0;
	volatile uint32_t dummy __attribute((unused)) = PORTB;
	IFS1CLR = _IFS1_CNBIF_MASK;
	IEC1SET = _IEC1_CNBIE_MASK;
	
	disableInterrupts();
	
	// initialize quadrature data
	_quadratureData = (PORTB >> 12) &0x03;
	
	// set interrupt priority for change notifications on port B
	// to highest priority -- we don't want to loose encoder pulses
	// this has effect of changing priority to ALL change bits
	IPC8bits.CNIP = 7;
	IPC8bits.CNIS = 0;

	setIntVector(_CHANGE_NOTICE_VECTOR, _quadratureISR);

	enableInterrupts();
#endif

	// initialize motor position
	_resetMotorPos();
}

// destructor
EncoderClass::~EncoderClass()
{
}

// set encoder placement
void EncoderClass::setBackPlacement(void)
{
	if(_placement)
		return;
	setOutputDirection0(_direction ? DIR_POS : DIR_NEG);
	_placement = true;
}
	
void EncoderClass::setFrontPlacement(void)
{
	if(!_placement)
		return;
	setOutputDirection0(_direction ? DIR_NEG : DIR_POS);
	_placement = false;
}

void EncoderClass::setPlacement(bool p)
{
	if(p)
		setBackPlacement();
	else
		setFrontPlacement();
}

// output direction - encoder placement dependent
void EncoderClass::setOutputDirectionCW(void)
{
	if(_direction)
		return;
	setOutputDirection0(_placement ? DIR_POS : DIR_NEG);
	_direction = true;
}

void EncoderClass::setOutputDirectionCCW(void)
{
	if(!_direction)
		return;
	setOutputDirection0(_placement ? DIR_NEG : DIR_POS);
	_direction = false;
}

void EncoderClass::setOutputDirection(bool dir)
{
	if(dir)
		setOutputDirectionCW();
	else
		setOutputDirectionCCW();
}

// read SSI absolute position with calibration data applied
// this one ensure that NO CONCURRENT hardware read is made
// if it can't read an updated position, just use previous one
uint16_t EncoderClass::readSSIPosition()
{
	// read raw SSI position (uncalibrated)
	uint16_t ssiPos = _readSSIPosition();
	
	// default to previous read if can't do now
	if(ssiPos == 0xffff)
		return _lastReadSSIPos;
	
	return ssiPos;
}

// get motor position (in stepper coordinates) - calibration data applied
int32_t EncoderClass::motorPos(void)
{
	uint16_t ssiPos;

#ifdef USE_QUADRATURE_SIGNALS
	// we can't trust quadrature motor position
	// wo we use it just as a rough position
	// and we correct it with SSI data on each query
	ssiPos = _readSSIPosition();

	// if failed reading SSI position, just return
	// uncorrected quadrature position
	if(ssiPos == 0xffff)
		return ENC2STEP(_motorPos);
	
	// we need shaft position from motor multi-turn position
	
	// fixup motor position from quadrature one
	int32_t mPos = _motorPos;
	
	// add zero position
	mPos += _zeroPos;

	// take modulo 4096
	int16_t mPos1 = mPos % 4096;
	if(mPos1 < 0)
		mPos1 += 4096;

	int32_t delta;
	bool err = false;
	if(mPos1 < ssiPos)
	{
		delta = ssiPos - mPos1;
		if(delta <= 1024)
		{
			_errorPrevMPos = _motorPos;
			_motorPos += delta;
		}
		else if(delta >= 3 * 1024)
		{
			_errorPrevMPos = _motorPos;
			_motorPos -= (4096 - delta);
		}
		else
			err = true;
	}
	else
	{
		delta = mPos1 - ssiPos;
		if(delta <= 1024)
		{
			_errorPrevMPos = _motorPos;
			_motorPos -= delta;
		}
		else if(delta >= 3 * 1024)
		{
			_errorPrevMPos = _motorPos;
			_motorPos += (4096 - delta);
		}
		else
			err = true;
	}

	if(err)
	{
		// signal the error and DON'T update the position
		_errorFlags |= (uint16_t)EncoderErrors::ERROR_TOO_BIG;
		_errorCurMPos = _motorPos;
		_errorMPOS_QUAD = mPos1;
		_error_MPOS_SSI = ssiPos;
	}


#else

	// if we don't use quadrature, we're forced to use SSI
	// we could force its reading, but this could also
	// make controller hang too much time.
	ssiPos = readSSIPosition();
	
	// calculate travel distance with a minimal error check
	// we shall call motorPos() routine fast enough
	// to avoid the one-turn overrun
	bool err = false;
	int32_t delta;
	if(_prevSSIPos < ssiPos)
	{
		delta = ssiPos - _prevSSIPos;
		if(delta <= 1024)
		{
			_errorPrevMPos = _motorPos;
			_motorPos += delta;
		}
		else if(delta >= 3 * 1024)
		{
			_errorPrevMPos = _motorPos;
			_motorPos -= (4096 - delta);
		}
		else
			err = true;
	}
	else
	{
		delta = _prevSSIPos - ssiPos;
		if(delta <= 1024)
		{
			_errorPrevMPos = _motorPos;
			_motorPos -= delta;
		}
		else if(delta >= 3 * 1024)
		{
			_errorPrevMPos = _motorPos;
			_motorPos += (4096 - delta);
		}
		else
			err = true;
	}
	if(err)
	{
		// signal the error and DON'T update the position
		_errorFlags |= EncoderErrors::ERROR_TOO_BIG;
		_errorCurMPos = _motorPos;
		_errorMPOS_QUAD = _prevSSIPos;
		_error_MPOS_SSI = ssiPos;
	}
	else
	{
		// store current SSI position
		_prevSSIPos = ssiPos;
	}
	
#endif

	// return motor position in stepper coordinates
	return ENC2STEP(_motorPos);
}

// enable/disable calibrated mode
void EncoderClass::enableCalibration(bool en)
{
	if(_hasCalibrationData)
		_calibrationEnabled = en;
	else
		_calibrationEnabled = false;
}

// dump calibration data on stream
#ifdef DEBUG
void EncoderClass::dumpCalibration(void)
{
	int8_t minData = 127, maxData = -127;
	DEBUG_INFO_N("MotorFish calibration data\n\n");
	for(int i = 0; i < 4096; i++)
	{
		if(!(i % 16))
			DEBUG_INFO_N("\nPOS:%05d ", i);
		int8_t d = _calibrationData[i];
		DEBUG_INFO_N("%+04d,", d);
		if(d < minData)
			minData = d;
		if(d > maxData)
			maxData = d; 
	}
	DEBUG_INFO_N("\nMIN:%+04d -- MAX:%+04d\n", minData, maxData);
	DEBUG_INFO_N("\nDONE\n");
}
#endif

// read error flags and various related data
uint16_t EncoderClass::errorFlags(void)
{
	return _errorFlags;
}

void EncoderClass::clearErrors(void)
{
	_errorFlags = 0;
	_errorSSI = 0xffff;
	_errorMPOS_QUAD = 0xffff;
	_error_MPOS_SSI = 0xffff;
}

void EncoderClass::printErrors(void)
{
	if(_errorFlags & (uint16_t)EncoderErrors::SSI_NOT_READY)
		DEBUG_ERROR("SSI value not ready\n");

	if(_errorFlags & (uint16_t)EncoderErrors::SSI_BAD_VALUE)
		DEBUG_ERROR("Bad value read from SSI interface:%05u\n", _errorSSI);

	if(_errorFlags & (uint16_t)EncoderErrors::ERROR_TOO_BIG)
#ifdef USE_QUADRATURE_SIGNALS
		DEBUG_ERROR("Too big distance from quadrature (%05u) and SSI (%05u) readings:\n", _errorMPOS_QUAD, _error_MPOS_SSI);
		DEBUG_ERROR("Previous MPOS:%07u - Current MPOS:%04u\n\n", _errorPrevMPos, _errorCurMPos);
#else
		DEBUG_ERROR("Too big distance from previous (%05u) and current (%05u) SSI readings:\n", _errorMPOS_QUAD, _error_MPOS_SSI);
		DEBUG_ERROR("Previous MPOS:%07u - Current MPOS:%04u\n\n", _errorPrevMPos, _errorCurMPos);
#endif
}

EncoderClass &__getEncoderClass(void)
{
	static EncoderClass encoder;
	return encoder;
}
