//////////////////////////////////////////////////////////////////////////////////////
//						 		Stepper.cpp											//
//																					//
//			Library to handle MotorFish closed loop stepper driver board			//
//																					//
//		Copyright (c) 2019 Massimo Del Fedele.  All rights reserved.				//
//																					//
//	Redistribution and use in source and binary forms, with or without				//
//	modification, are permitted provided that the following conditions are met:		//
//																					//
//	- Redistributions of source code must retain the above copyright notice,		//
//	  this list of conditions and the following disclaimer.							//
//	- Redistributions in binary form must reproduce the above copyright notice,		//
//	  this list of conditions and the following disclaimer in the documentation		//
//	  and/or other materials provided with the distribution.						//
//																					//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"		//
//	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE		//
//	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE		//
//	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE		//
//	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR				//
//	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF			//
//	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS		//
//	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN			//
//	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)			//
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE		//
//	POSSIBILITY OF SUCH DAMAGE.														//
//																					//
//  Version 7.8.0 -- March 2019		Initial version									//
//	Version 8.0.0 - 26/07/2020 - UPDATED FOR FIRMWARE 8.0.0							//
//																					//
//////////////////////////////////////////////////////////////////////////////////////
#include "Stepper.h"
#include "Tables.h"

#define DEBUG_LEVEL_INFO
#include <FishinoDebug.h>

//////////////////////////////////////////////////////////////////
//				MotorFish driver connections					//
//////////////////////////////////////////////////////////////////

// First winding
const int AVREF		= 17;	// pwm reference voltage
const int AIN1		= 18;	// control input 1
const int AIN2		= 19;	// control input 2
const int AI0		= 20;	// current scale input 1
const int AI1		= 21;	// current scale input 2

// second winding
const int BVREF		= 22;	// pwm reference voltage
const int BIN1		= 23;	// control input 1
const int BIN2		= 24;	// control input 2
const int BI0		= 25;	// current scale input 1
const int BI1		= 26;	// current scale input 2

// control I/Os
const int NRESET	= 27;	// reset input (active low)
const int NSLEEP	= 28;	// sleep input (active low)
const int NFAULT	= 29;	// error output (active low)
const int DECAY		= 30;	// decay mode selecton (3 state)

const double RSense	= 0.20;	// current sense resistor value (ohm)

// the motor positive direction (true = CW, false = CCW)
volatile bool StepperClass::_motorDirection = true;

// microstep phase
volatile uint16_t StepperClass::_phase = 0;

// winding current (nominal)
volatile double StepperClass::_windingCurrent = 1.0;

// winding current factor
// (used to reduce from nominal, as 1/65536 multiplicator)
volatile uint16_t StepperClass::_windingCurrentFactor = 65535;

// pwm full scale value for requested winding current
volatile uint16_t StepperClass::_pwmWindingCurrent = 0;

// energized flag
volatile bool StepperClass::_energized = false;

// constructor
StepperClass::StepperClass()
{
	_phase = 0;
	_energized = false;
	
	analogWriteFrequency(200000);
	analogWriteResolution(16);

	// default to a safe 0.5 Ampere current value
	setWindingCurrent(0.5);
	setWindingCurrentFactor(65535);

	// motor direction = CW
	setDirectionCW();

	// keep driver on reset
	pinMode(NRESET, OUTPUT);
	digitalWrite(NRESET, LOW);
	
	// disable sleep mode
	pinMode(NSLEEP, OUTPUT);
	digitalWrite(NSLEEP, HIGH);
	
	// put fault pin as input
	pinMode(NFAULT, INPUT_PULLUP);
	
	// select mixed-mode decay
//	pinMode(DECAY, INPUT);
pinMode(DECAY, OUTPUT);
digitalWrite(DECAY, HIGH);
	
	// shut down motors
	pinMode(AIN1, OUTPUT);
	digitalWrite(AIN1, 0);
	pinMode(AIN2, OUTPUT);
	digitalWrite(AIN2, 0);
	pinMode(BIN1, OUTPUT);
	digitalWrite(BIN1, 0);
	pinMode(BIN2, OUTPUT);
	digitalWrite(BIN2, 0);
	
	// select full current output for both windings
	pinMode(AI0, OUTPUT);
	digitalWrite(AI0, 0);
	pinMode(AI1, OUTPUT);
	digitalWrite(AI1, 0);
	pinMode(BI0, OUTPUT);
	digitalWrite(BI0, 0);
	pinMode(BI1, OUTPUT);
	digitalWrite(BI1, 0);
	
	pinMode(AVREF, OUTPUT);
	analogWrite(AVREF, 0);
	pinMode(BVREF, OUTPUT);
	analogWrite(BVREF, 0);

	// release driver reset
	digitalWrite(NRESET, HIGH);
}

// destructor
StepperClass::~StepperClass()
{
}

// step forward
void StepperClass::stepForward(void)
{
	// do not move if not energized
	if(!_energized)
		return;
	
	_phase = (_phase + 1) & 0xff;
	energize(true);
}

// step backwards
void StepperClass::stepBackward(void)
{
	// do not move if not energized
	if(!_energized)
		return;
	
	_phase = (_phase - 1) & 0xff;
	energize(true);
}
			
// energize/deenergize the motor
void StepperClass::energize(bool en)
{
	uint32_t aPwm, bPwm;
	
	_energized = en;
	if(en)
	{
		// apply current factor
		uint32_t current = (uint32_t)_pwmWindingCurrent * (uint32_t)_windingCurrentFactor / 65535;

		uint8_t aPhase = APHASE(_phase);
		aPwm = (uint32_t)COSINETABLE(_phase) * (uint32_t)current;
		aPwm >>= 16;

		// if we want CCW positive rotation, just invert ONE
		// motor phase
		uint8_t bPhase = BPHASE(_phase);
		if(!_motorDirection)
			bPhase = 1 - bPhase;
		
		bPwm = (uint32_t)SINETABLE(_phase) * (uint32_t)current;
		bPwm >>= 16;
		
//		DEBUG_INFO("A%c%05d B%c%05d\n", aPhase ? '>':'<', aPwm, bPhase ? '>':'<', bPwm);

		digitalWrite(AIN1, aPhase);
		digitalWrite(AIN2, !aPhase);

		digitalWrite(BIN1, bPhase);
		digitalWrite(BIN2, !bPhase);
		
		analogWrite(AVREF, aPwm);
		analogWrite(BVREF, bPwm);

//		DEBUG_INFO("phase:%4d  aPhase:%1d  aPwm:%6d  bPhase:%1d, bPwm:%6d\n", _phase, aPhase, aPwm, bPhase, bPwm);
	}
	else
	{
		analogWrite(AVREF, 0);
		digitalWrite(AIN1, LOW);
		digitalWrite(AIN2, LOW);

		analogWrite(BVREF, 0);
		digitalWrite(BIN1, LOW);
		digitalWrite(BIN2, LOW);
	}
}

// set winding current
void StepperClass::setWindingCurrent(double c)
{
	_windingCurrent = c;
	
	// 5.0 is a driver constant, 3.3V is pwm full output voltage
	double ARef = 5.0 * RSense * c / 3.3;
	_pwmWindingCurrent = (uint16_t)(ARef * 65535);

	// re-energize the motor as needed
	energize(_energized);
}

// set winding current reduction factor (as 1/65535 of this value)
void StepperClass::setWindingCurrentFactor(uint16_t f)
{
	_windingCurrentFactor = f;

	// re-energize the motor as needed
	energize(_energized);
}

// set motor phase and current reduction factor
// used on closed loop control
void StepperClass::setPhaseFactor(uint16_t phase, uint16_t factor)
{
#ifdef USE_ENCODER_TABLE
	_phase = phase & 0x7ff;
#else
	_phase = phase & 0xff;
#endif
	_windingCurrentFactor = factor;

	// re-energize the motor as needed
	energize(_energized);
}

// check error condition
bool StepperClass::isError(void)
{
	return !digitalRead(NFAULT);
}


StepperClass &__getStepperClass(void)
{
	static StepperClass stepper;
	return stepper;
}
