//////////////////////////////////////////////////////////////////////////////////////
//						 		Stepper.h											//
//																					//
//			Library to handle MotorFish closed loop stepper driver board			//
//																					//
//		Copyright (c) 2019 Massimo Del Fedele.  All rights reserved.				//
//																					//
//	Redistribution and use in source and binary forms, with or without				//
//	modification, are permitted provided that the following conditions are met:		//
//																					//
//	- Redistributions of source code must retain the above copyright notice,		//
//	  this list of conditions and the following disclaimer.							//
//	- Redistributions in binary form must reproduce the above copyright notice,		//
//	  this list of conditions and the following disclaimer in the documentation		//
//	  and/or other materials provided with the distribution.						//
//																					//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"		//
//	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE		//
//	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE		//
//	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE		//
//	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR				//
//	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF			//
//	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS		//
//	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN			//
//	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)			//
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE		//
//	POSSIBILITY OF SUCH DAMAGE.														//
//																					//
//  Version 7.8.0 -- March 2019		Initial version									//
//	Version 8.0.0 - 26/07/2020 - UPDATED FOR FIRMWARE 8.0.0							//
//																					//
//////////////////////////////////////////////////////////////////////////////////////
#ifndef __STEPPER_H
#define __STEPPER_H

#include <Arduino.h>

class StepperClass
{
	friend StepperClass &__getStepperClass(void);
	
	private:
		
		// the motor positive direction (true = CW, false = CCW)
		volatile static bool _motorDirection;

		// microstep phase
		volatile static uint16_t _phase;
		
		// winding current (nominal)
		volatile static double _windingCurrent;
		
		// winding current factor
		// (used to reduce from nominal, as 1/65536 multiplicator)
		volatile static uint16_t _windingCurrentFactor;
		
		// pwm full scale value for requested winding current
		volatile static uint16_t _pwmWindingCurrent;
		
		// energized flag
		volatile static bool _energized;
		
		// constructor
		StepperClass();

	protected:

	public:

		// destructor
		~StepperClass();
		
		// set motor direction
		static void setDirection(bool d)
			{ _motorDirection = d; }
		
		// set motor positive direction
		static void setDirectionCW(void)
			{ _motorDirection = true; }
			
		static void setDirectionCCW(void)
			{ _motorDirection = false; }
			
		// get motor positive direction
		static bool isDirectionCW(void)
			{ return _motorDirection; }
			
		static bool isDirectionCCW(void)
			{ return !_motorDirection; }
		
		// step
		inline static void step(bool fwd = true)
			{ if(fwd) stepForward(); else stepBackward(); }
		
		// step forward
		static void stepForward(void);
		
		// step backwards
		static void stepBackward(void);
			
		// energize/deenergize the motor
		static void energize(bool en);
		
		// check energized state
		static bool isEnergized(void)
			{ return _energized; }
		
		// set winding current
		static void setWindingCurrent(double c);
		
		// get winding current
		static double getWindingCurrent(void)
			{ return _windingCurrent; }
		
		// set winding current reduction factor (as 1/65535 of this value)
		static void setWindingCurrentFactor(uint16_t f);
		
		// get winding current factor
		static uint16_t getWindingCurrentFactor(void)
			{ return _windingCurrentFactor; }
		
		// set motor phase and current reduction factor
		// used on closed loop control
		static void setPhaseFactor(uint16_t phase, uint16_t factor);
		
		// get current phase
		static uint16_t getPhase(void)
			{ return _phase; }

		// check error condition
		static bool isError(void);
};

#define Stepper __getStepperClass()

extern StepperClass &__getStepperClass(void);

#endif
